<?php
/**
 * Laravel版: PostController.php
 * app/Http/Controllers/PostController.php
 * 
 * Laravelの利点：
 * - コントローラーとビューの責任分離
 * - 自動的なCSRF保護
 * - 洗練されたバリデーション機能
 * - 一貫性のあるエラーハンドリング
 */

namespace App\Http\Controllers;

use App\Models\Post;
use Illuminate\Http\Request;

class PostController extends Controller
{
    /**
     * 投稿一覧と投稿フォームを表示
     */
    public function index()
    {
        // Eloquent ORMで簡潔にデータ取得
        $posts = Post::orderBy('created_at', 'desc')->get();
        
        // ビューにデータを渡す（自動的にエスケープされる）
        return view('posts.index', compact('posts'));
    }
    
    /**
     * 新規投稿を保存
     */
    public function store(Request $request)
    {
        // バリデーション（自動的にエラー処理とリダイレクト）
        $validated = $request->validate([
            'name' => 'required|max:50',
            'message' => 'required|max:1000'
        ], [
            'name.required' => '名前を入力してください。',
            'name.max' => '名前は50文字以内で入力してください。',
            'message.required' => 'メッセージを入力してください。',
            'message.max' => 'メッセージは1000文字以内で入力してください。'
        ]);
        
        // Mass Assignment保護付きでデータ保存
        Post::create($validated);
        
        // フラッシュメッセージ付きでリダイレクト
        return redirect()->route('posts.index')
            ->with('success', '投稿が完了しました。');
    }
}