# 第3章 サンプルコード - ルーティングとMVCの基礎

このディレクトリには、第3章で解説するMVCアーキテクチャとルーティングのサンプルコードが含まれています。

## ディレクトリ構成

```
chapter03/
├── routes/
│   ├── web.php              # 基本的なルート定義
│   └── api.php              # API用ルート定義
├── controllers/
│   ├── ProductController.php # RESTfulコントローラー
│   └── UserController.php    # 基本的なコントローラー
├── models/
│   └── Product.php          # Eloquentモデル
└── views/
    ├── layouts/
    │   └── app.blade.php    # レイアウトテンプレート
    ├── products/
    │   ├── index.blade.php  # 商品一覧
    │   ├── show.blade.php   # 商品詳細
    │   └── create.blade.php # 商品作成フォーム
    └── components/
        └── alert.blade.php  # 再利用可能なコンポーネント
```

## MVCアーキテクチャの理解

### Model（モデル）
- データベースとのやり取りを担当
- ビジネスロジックを実装
- Eloquent ORMを使用

### View（ビュー）
- ユーザーインターフェースを担当
- Bladeテンプレートエンジンを使用
- HTMLの生成と表示

### Controller（コントローラー）
- ModelとViewの橋渡し役
- リクエストの処理とレスポンスの生成
- ビジネスロジックの調整

## ルーティングの基本

### 基本的なルート定義

```php
// GET リクエスト
Route::get('/products', [ProductController::class, 'index']);

// POST リクエスト
Route::post('/products', [ProductController::class, 'store']);

// PUT/PATCH リクエスト
Route::put('/products/{id}', [ProductController::class, 'update']);

// DELETE リクエスト
Route::delete('/products/{id}', [ProductController::class, 'destroy']);
```

### RESTfulリソースルート

```php
// 7つの基本アクションを一度に定義
Route::resource('products', ProductController::class);
```

生成されるルート：
- GET `/products` - 一覧表示 (index)
- GET `/products/create` - 作成フォーム (create)
- POST `/products` - 保存処理 (store)
- GET `/products/{id}` - 詳細表示 (show)
- GET `/products/{id}/edit` - 編集フォーム (edit)
- PUT/PATCH `/products/{id}` - 更新処理 (update)
- DELETE `/products/{id}` - 削除処理 (destroy)

## 実装のポイント

### 1. ルートパラメータのバリデーション

```php
Route::get('/users/{id}', function ($id) {
    // 処理
})->where('id', '[0-9]+');
```

### 2. ルートグループ

```php
Route::prefix('admin')->group(function () {
    Route::get('/users', [AdminController::class, 'users']);
    Route::get('/products', [AdminController::class, 'products']);
});
```

### 3. ミドルウェアの適用

```php
Route::middleware(['auth'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index']);
});
```

## Bladeテンプレートの活用

### テンプレート継承

```blade
{{-- layouts/app.blade.php --}}
<!DOCTYPE html>
<html>
<head>
    <title>@yield('title')</title>
</head>
<body>
    @yield('content')
</body>
</html>

{{-- products/index.blade.php --}}
@extends('layouts.app')

@section('title', '商品一覧')

@section('content')
    <h1>商品一覧</h1>
    {{-- コンテンツ --}}
@endsection
```

### 条件分岐とループ

```blade
@if($products->count() > 0)
    @foreach($products as $product)
        <div>{{ $product->name }}</div>
    @endforeach
@else
    <p>商品がありません</p>
@endif
```

## 実行方法

1. Laravelプロジェクトにサンプルファイルをコピー
2. データベースマイグレーションを実行
3. 開発サーバーを起動：`php artisan serve`
4. ブラウザでアクセス：`http://localhost:8000`

## 学習のポイント

1. **責任の分離**
   - 各コンポーネントの役割を明確に理解
   - 適切な場所に適切なコードを配置

2. **RESTful設計**
   - HTTPメソッドの適切な使用
   - 統一的なURL設計

3. **テンプレートの再利用**
   - レイアウトの共通化
   - コンポーネントの分離

4. **セキュリティ**
   - CSRF保護の自動適用
   - XSS対策の自動エスケープ