<?php
/**
 * app/Http/Controllers/ProductController.php
 * 
 * 商品管理のRESTfulコントローラー
 * CRUD操作の標準的な実装例
 */

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    /**
     * 商品一覧を表示
     * 
     * GET /products
     */
    public function index(Request $request)
    {
        // クエリパラメータで検索・ソート機能を実装
        $query = Product::query();
        
        // 検索機能
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
        }
        
        // カテゴリフィルター
        if ($request->has('category')) {
            $query->where('category_id', $request->input('category'));
        }
        
        // ソート機能
        $sortBy = $request->input('sort', 'created_at');
        $sortOrder = $request->input('order', 'desc');
        $query->orderBy($sortBy, $sortOrder);
        
        // ページネーション（1ページ12件）
        $products = $query->paginate(12);
        
        // ビューにデータを渡す
        return view('products.index', [
            'products' => $products,
            'search' => $request->input('search'),
            'currentCategory' => $request->input('category')
        ]);
    }
    
    /**
     * 商品作成フォームを表示
     * 
     * GET /products/create
     */
    public function create()
    {
        // カテゴリー一覧を取得（フォームのセレクトボックス用）
        $categories = \App\Models\Category::all();
        
        return view('products.create', compact('categories'));
    }
    
    /**
     * 新規商品を保存
     * 
     * POST /products
     */
    public function store(Request $request)
    {
        // バリデーション
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'image' => 'nullable|image|max:2048' // 2MBまで
        ], [
            'name.required' => '商品名は必須です。',
            'name.max' => '商品名は255文字以内で入力してください。',
            'description.required' => '商品説明は必須です。',
            'price.required' => '価格は必須です。',
            'price.numeric' => '価格は数値で入力してください。',
            'price.min' => '価格は0以上で入力してください。',
            'stock.required' => '在庫数は必須です。',
            'stock.integer' => '在庫数は整数で入力してください。',
            'stock.min' => '在庫数は0以上で入力してください。',
            'category_id.required' => 'カテゴリーを選択してください。',
            'category_id.exists' => '選択されたカテゴリーが存在しません。',
            'image.image' => '画像ファイルをアップロードしてください。',
            'image.max' => '画像は2MB以下にしてください。'
        ]);
        
        // トランザクション処理
        DB::beginTransaction();
        try {
            // 商品データを保存
            $product = new Product();
            $product->fill($validated);
            
            // 画像のアップロード処理
            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('products', 'public');
                $product->image_path = $path;
            }
            
            $product->save();
            
            DB::commit();
            
            // 成功メッセージと共にリダイレクト
            return redirect()
                ->route('products.show', $product)
                ->with('success', '商品を登録しました。');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            // エラーメッセージと共に元のフォームへ
            return back()
                ->withInput()
                ->with('error', '商品の登録に失敗しました。');
        }
    }
    
    /**
     * 商品詳細を表示
     * 
     * GET /products/{product}
     */
    public function show(Product $product)
    {
        // 関連データを事前読み込み（N+1問題対策）
        $product->load(['category', 'reviews.user']);
        
        // 関連商品を取得
        $relatedProducts = Product::where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->limit(4)
            ->get();
        
        return view('products.show', [
            'product' => $product,
            'relatedProducts' => $relatedProducts
        ]);
    }
    
    /**
     * 商品編集フォームを表示
     * 
     * GET /products/{product}/edit
     */
    public function edit(Product $product)
    {
        // 編集権限のチェック（ポリシーを使用）
        $this->authorize('update', $product);
        
        $categories = \App\Models\Category::all();
        
        return view('products.edit', [
            'product' => $product,
            'categories' => $categories
        ]);
    }
    
    /**
     * 商品情報を更新
     * 
     * PUT/PATCH /products/{product}
     */
    public function update(Request $request, Product $product)
    {
        // 編集権限のチェック
        $this->authorize('update', $product);
        
        // バリデーション
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'image' => 'nullable|image|max:2048'
        ]);
        
        DB::beginTransaction();
        try {
            // 商品データを更新
            $product->fill($validated);
            
            // 新しい画像がアップロードされた場合
            if ($request->hasFile('image')) {
                // 古い画像を削除
                if ($product->image_path) {
                    \Storage::disk('public')->delete($product->image_path);
                }
                
                // 新しい画像を保存
                $path = $request->file('image')->store('products', 'public');
                $product->image_path = $path;
            }
            
            $product->save();
            
            DB::commit();
            
            return redirect()
                ->route('products.show', $product)
                ->with('success', '商品情報を更新しました。');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return back()
                ->withInput()
                ->with('error', '商品の更新に失敗しました。');
        }
    }
    
    /**
     * 商品を削除
     * 
     * DELETE /products/{product}
     */
    public function destroy(Product $product)
    {
        // 削除権限のチェック
        $this->authorize('delete', $product);
        
        DB::beginTransaction();
        try {
            // 画像ファイルを削除
            if ($product->image_path) {
                \Storage::disk('public')->delete($product->image_path);
            }
            
            // 商品データを削除
            $product->delete();
            
            DB::commit();
            
            return redirect()
                ->route('products.index')
                ->with('success', '商品を削除しました。');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return back()
                ->with('error', '商品の削除に失敗しました。');
        }
    }
}