<?php
/**
 * app/Models/Product.php
 * 
 * 商品モデル
 * Eloquent ORMを使用したデータベース操作の実装例
 */

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * テーブル名の指定（省略可能：クラス名の複数形が自動的に使用される）
     */
    protected $table = 'products';
    
    /**
     * 主キーの指定（省略可能：'id'がデフォルト）
     */
    protected $primaryKey = 'id';
    
    /**
     * 主キーの型（省略可能：'int'がデフォルト）
     */
    protected $keyType = 'int';
    
    /**
     * タイムスタンプの自動管理（省略可能：trueがデフォルト）
     */
    public $timestamps = true;
    
    /**
     * Mass Assignment保護
     * fillableに指定したカラムのみ一括代入可能
     */
    protected $fillable = [
        'name',
        'description',
        'price',
        'stock',
        'category_id',
        'image_path',
        'is_featured',
        'published_at'
    ];
    
    /**
     * 一括代入を禁止するカラム（fillableの代わりに使用可能）
     */
    // protected $guarded = ['id', 'created_at', 'updated_at'];
    
    /**
     * 日付として扱うカラム
     */
    protected $dates = [
        'published_at',
        'deleted_at'
    ];
    
    /**
     * キャストする属性
     */
    protected $casts = [
        'price' => 'decimal:2',
        'stock' => 'integer',
        'is_featured' => 'boolean',
        'published_at' => 'datetime',
        'metadata' => 'array', // JSON型カラムを配列として扱う
    ];
    
    /**
     * デフォルト値
     */
    protected $attributes = [
        'stock' => 0,
        'is_featured' => false,
    ];
    
    /**
     * 配列/JSONに含める属性
     */
    protected $visible = [
        'id',
        'name',
        'description',
        'price',
        'stock',
        'formatted_price',
        'availability_status'
    ];
    
    /**
     * 配列/JSONから除外する属性
     */
    protected $hidden = [
        'deleted_at',
        'internal_notes'
    ];
    
    /**
     * 追加する属性（アクセサ）
     */
    protected $appends = [
        'formatted_price',
        'availability_status'
    ];
    
    /*
    |--------------------------------------------------------------------------
    | リレーション定義
    |--------------------------------------------------------------------------
    */
    
    /**
     * カテゴリーとのリレーション（多対1）
     */
    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    
    /**
     * レビューとのリレーション（1対多）
     */
    public function reviews()
    {
        return $this->hasMany(Review::class);
    }
    
    /**
     * タグとのリレーション（多対多）
     */
    public function tags()
    {
        return $this->belongsToMany(Tag::class)
            ->withTimestamps()
            ->withPivot('sort_order');
    }
    
    /**
     * 注文明細とのリレーション（1対多）
     */
    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }
    
    /*
    |--------------------------------------------------------------------------
    | アクセサ（Getter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * フォーマット済み価格を取得
     */
    public function getFormattedPriceAttribute()
    {
        return '¥' . number_format($this->price);
    }
    
    /**
     * 在庫状況を取得
     */
    public function getAvailabilityStatusAttribute()
    {
        if ($this->stock > 10) {
            return '在庫あり';
        } elseif ($this->stock > 0) {
            return '残りわずか';
        } else {
            return '在庫なし';
        }
    }
    
    /**
     * 商品URLを取得
     */
    public function getUrlAttribute()
    {
        return route('products.show', $this);
    }
    
    /**
     * 画像URLを取得
     */
    public function getImageUrlAttribute()
    {
        if ($this->image_path) {
            return asset('storage/' . $this->image_path);
        }
        return asset('images/no-image.jpg');
    }
    
    /*
    |--------------------------------------------------------------------------
    | ミューテータ（Setter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * 価格を設定する際の処理
     */
    public function setPriceAttribute($value)
    {
        // 消費税込み価格を自動計算（例）
        $this->attributes['price'] = $value;
        $this->attributes['price_with_tax'] = $value * 1.1;
    }
    
    /**
     * 商品名を設定する際の処理
     */
    public function setNameAttribute($value)
    {
        // 商品名を適切にフォーマット
        $this->attributes['name'] = trim($value);
        // スラッグを自動生成
        $this->attributes['slug'] = \Str::slug($value);
    }
    
    /*
    |--------------------------------------------------------------------------
    | スコープ（クエリの再利用）
    |--------------------------------------------------------------------------
    */
    
    /**
     * 公開済み商品のみ取得
     */
    public function scopePublished($query)
    {
        return $query->where('published_at', '<=', now())
                     ->whereNull('deleted_at');
    }
    
    /**
     * 在庫がある商品のみ取得
     */
    public function scopeInStock($query)
    {
        return $query->where('stock', '>', 0);
    }
    
    /**
     * 特集商品のみ取得
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
    
    /**
     * 価格範囲で絞り込み
     */
    public function scopePriceRange($query, $min, $max)
    {
        return $query->whereBetween('price', [$min, $max]);
    }
    
    /**
     * カテゴリーで絞り込み
     */
    public function scopeInCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }
    
    /*
    |--------------------------------------------------------------------------
    | ビジネスロジック
    |--------------------------------------------------------------------------
    */
    
    /**
     * 在庫を減らす
     */
    public function decrementStock($quantity = 1)
    {
        if ($this->stock < $quantity) {
            throw new \Exception('在庫が不足しています。');
        }
        
        $this->decrement('stock', $quantity);
    }
    
    /**
     * 在庫を増やす
     */
    public function incrementStock($quantity = 1)
    {
        $this->increment('stock', $quantity);
    }
    
    /**
     * 平均評価を計算
     */
    public function calculateAverageRating()
    {
        return $this->reviews()->avg('rating') ?? 0;
    }
    
    /**
     * 商品が購入可能かチェック
     */
    public function isAvailable()
    {
        return $this->stock > 0 && 
               $this->published_at <= now() &&
               !$this->deleted_at;
    }
    
    /**
     * 関連商品を取得
     */
    public function getRelatedProducts($limit = 4)
    {
        return self::where('category_id', $this->category_id)
            ->where('id', '!=', $this->id)
            ->published()
            ->inStock()
            ->limit($limit)
            ->get();
    }
    
    /*
    |--------------------------------------------------------------------------
    | イベント
    |--------------------------------------------------------------------------
    */
    
    /**
     * モデルイベントの登録
     */
    protected static function booted()
    {
        // 作成前の処理
        static::creating(function ($product) {
            // SKUを自動生成
            if (empty($product->sku)) {
                $product->sku = 'PRD-' . strtoupper(uniqid());
            }
        });
        
        // 削除前の処理
        static::deleting(function ($product) {
            // 関連する画像を削除
            if ($product->image_path) {
                \Storage::disk('public')->delete($product->image_path);
            }
        });
    }
}