<?php
/**
 * routes/web.php
 * 
 * Webアプリケーションのルート定義ファイル
 * URLとコントローラーアクションのマッピングを管理
 */

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\UserController;

/*
|--------------------------------------------------------------------------
| 基本的なルート定義
|--------------------------------------------------------------------------
*/

// トップページ
Route::get('/', function () {
    return view('welcome');
})->name('home');

// 静的ページ
Route::get('/about', function () {
    return view('about');
})->name('about');

/*
|--------------------------------------------------------------------------
| 商品管理システム - RESTfulルート
|--------------------------------------------------------------------------
*/

// 個別のルート定義方法
Route::get('/products', [ProductController::class, 'index'])->name('products.index');
Route::get('/products/create', [ProductController::class, 'create'])->name('products.create');
Route::post('/products', [ProductController::class, 'store'])->name('products.store');
Route::get('/products/{product}', [ProductController::class, 'show'])->name('products.show');
Route::get('/products/{product}/edit', [ProductController::class, 'edit'])->name('products.edit');
Route::put('/products/{product}', [ProductController::class, 'update'])->name('products.update');
Route::delete('/products/{product}', [ProductController::class, 'destroy'])->name('products.destroy');

// または、リソースルートで一括定義（上記と同等）
// Route::resource('products', ProductController::class);

/*
|--------------------------------------------------------------------------
| パラメータ付きルート
|--------------------------------------------------------------------------
*/

// 基本的なパラメータ
Route::get('/users/{id}', [UserController::class, 'show'])
    ->name('users.show');

// パラメータのバリデーション（数値のみ）
Route::get('/posts/{id}', function ($id) {
    return "Post ID: {$id}";
})->where('id', '[0-9]+');

// 複数パラメータのバリデーション
Route::get('/posts/{year}/{month}', function ($year, $month) {
    return "Archive: {$year}/{$month}";
})->where(['year' => '[0-9]{4}', 'month' => '[0-9]{2}']);

// オプショナルパラメータ
Route::get('/search/{keyword?}', function ($keyword = null) {
    if ($keyword) {
        return "検索キーワード: {$keyword}";
    }
    return "検索画面";
});

/*
|--------------------------------------------------------------------------
| ルートグループ
|--------------------------------------------------------------------------
*/

// プレフィックス付きグループ
Route::prefix('admin')->name('admin.')->group(function () {
    Route::get('/dashboard', function () {
        return view('admin.dashboard');
    })->name('dashboard');
    
    Route::get('/users', function () {
        return view('admin.users');
    })->name('users');
    
    Route::get('/settings', function () {
        return view('admin.settings');
    })->name('settings');
});

// ミドルウェア付きグループ（認証が必要なルート）
Route::middleware(['auth'])->group(function () {
    Route::get('/profile', [UserController::class, 'profile'])->name('profile');
    Route::put('/profile', [UserController::class, 'updateProfile'])->name('profile.update');
    Route::get('/orders', function () {
        return view('orders.index');
    })->name('orders');
});

// 複数の属性を持つグループ
Route::middleware(['auth', 'verified'])
    ->prefix('account')
    ->name('account.')
    ->group(function () {
        Route::get('/settings', function () {
            return view('account.settings');
        })->name('settings');
        
        Route::get('/billing', function () {
            return view('account.billing');
        })->name('billing');
    });

/*
|--------------------------------------------------------------------------
| 特殊なルート
|--------------------------------------------------------------------------
*/

// リダイレクト
Route::redirect('/old-page', '/new-page', 301);

// ビューを直接返す
Route::view('/privacy', 'legal.privacy')->name('privacy');
Route::view('/terms', 'legal.terms')->name('terms');

// フォールバックルート（404ページ）
Route::fallback(function () {
    return view('errors.404');
});

/*
|--------------------------------------------------------------------------
| ルートモデルバインディング
|--------------------------------------------------------------------------
*/

// 暗黙的バインディング
// URLの{product}パラメータが自動的にProductモデルのインスタンスになる
Route::get('/catalog/{product}', function (\App\Models\Product $product) {
    return view('catalog.show', compact('product'));
});

// カスタムキーを使用したバインディング
Route::get('/catalog/{product:slug}', function (\App\Models\Product $product) {
    return view('catalog.show', compact('product'));
});

/*
|--------------------------------------------------------------------------
| レート制限
|--------------------------------------------------------------------------
*/

// APIのレート制限（1分間に60回まで）
Route::middleware('throttle:60,1')->group(function () {
    Route::post('/api/search', function () {
        // 検索処理
    });
});

// カスタムレート制限
Route::middleware('throttle:uploads')->group(function () {
    Route::post('/upload', function () {
        // アップロード処理
    });
});