<?php
/**
 * database/migrations/2024_01_01_000002_create_posts_table.php
 * 
 * 投稿テーブルのマイグレーション
 * 外部キー制約を含むテーブル作成の例
 */

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * マイグレーション実行
     */
    public function up(): void
    {
        Schema::create('posts', function (Blueprint $table) {
            // 主キー
            $table->id();
            
            // 外部キー（ユーザーID）
            $table->foreignId('user_id')
                ->constrained()                    // usersテーブルのidを参照
                ->onDelete('cascade')              // ユーザー削除時に投稿も削除
                ->onUpdate('cascade');              // ユーザーID更新時に追従
            
            // 基本情報
            $table->string('title');                      // タイトル
            $table->string('slug')->unique();             // URLスラッグ（SEO用）
            $table->text('content');                      // 本文
            $table->text('excerpt')->nullable();          // 抜粋
            
            // メタ情報
            $table->string('featured_image')->nullable(); // アイキャッチ画像
            $table->json('meta_data')->nullable();        // メタデータ（JSON形式）
            
            // ステータス
            $table->enum('status', ['draft', 'published', 'private', 'archived'])
                ->default('draft');                       // 公開状態
            $table->boolean('is_featured')->default(false); // おすすめフラグ
            $table->boolean('allow_comments')->default(true); // コメント許可
            
            // カウンター
            $table->unsignedInteger('view_count')->default(0);    // 閲覧数
            $table->unsignedInteger('like_count')->default(0);    // いいね数
            $table->unsignedInteger('comment_count')->default(0); // コメント数
            
            // 公開日時
            $table->timestamp('published_at')->nullable(); // 公開日時
            
            // タイムスタンプ
            $table->timestamps();
            
            // ソフトデリート
            $table->softDeletes();
            
            // インデックス
            $table->index('user_id');
            $table->index('slug');
            $table->index('status');
            $table->index('published_at');
            $table->index(['status', 'published_at']);    // 複合インデックス
            $table->fullText('title');                    // 全文検索用
            $table->fullText('content');
        });
    }
    
    /**
     * マイグレーションのロールバック
     */
    public function down(): void
    {
        Schema::dropIfExists('posts');
    }
};