<?php
/**
 * app/Models/User.php
 * 
 * ユーザーモデル
 * リレーション、スコープ、アクセサ/ミューテータの実装例
 */

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable implements MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;
    
    /**
     * Mass Assignment可能な属性
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'avatar',
        'bio',
        'birth_date',
        'gender',
        'phone',
        'address',
        'city',
        'postal_code',
        'is_active',
        'is_admin',
        'last_login_at'
    ];
    
    /**
     * 配列/JSONから隠す属性
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];
    
    /**
     * キャストする属性
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'last_login_at' => 'datetime',
        'birth_date' => 'date',
        'is_active' => 'boolean',
        'is_admin' => 'boolean',
        'login_count' => 'integer',
    ];
    
    /**
     * アクセサに追加する属性
     */
    protected $appends = [
        'full_address',
        'age',
        'avatar_url'
    ];
    
    /*
    |--------------------------------------------------------------------------
    | リレーション
    |--------------------------------------------------------------------------
    */
    
    /**
     * ユーザーの投稿（1対多）
     */
    public function posts()
    {
        return $this->hasMany(Post::class);
    }
    
    /**
     * ユーザーのコメント（1対多）
     */
    public function comments()
    {
        return $this->hasMany(Comment::class);
    }
    
    /**
     * フォローしているユーザー（多対多）
     */
    public function following()
    {
        return $this->belongsToMany(User::class, 'follows', 'follower_id', 'following_id')
            ->withTimestamps()
            ->withPivot('created_at');
    }
    
    /**
     * フォロワー（多対多）
     */
    public function followers()
    {
        return $this->belongsToMany(User::class, 'follows', 'following_id', 'follower_id')
            ->withTimestamps()
            ->withPivot('created_at');
    }
    
    /**
     * いいねした投稿（多対多）
     */
    public function likedPosts()
    {
        return $this->belongsToMany(Post::class, 'likes')
            ->withTimestamps();
    }
    
    /*
    |--------------------------------------------------------------------------
    | アクセサ（Getter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * フルアドレスを取得
     */
    public function getFullAddressAttribute()
    {
        $parts = array_filter([
            $this->postal_code,
            $this->city,
            $this->address
        ]);
        
        return implode(' ', $parts);
    }
    
    /**
     * 年齢を計算
     */
    public function getAgeAttribute()
    {
        if (!$this->birth_date) {
            return null;
        }
        
        return $this->birth_date->age;
    }
    
    /**
     * アバターURLを取得
     */
    public function getAvatarUrlAttribute()
    {
        if ($this->avatar) {
            return asset('storage/' . $this->avatar);
        }
        
        // Gravatarを使用
        $hash = md5(strtolower(trim($this->email)));
        return "https://www.gravatar.com/avatar/{$hash}?d=mp";
    }
    
    /*
    |--------------------------------------------------------------------------
    | ミューテータ（Setter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * パスワードを自動的にハッシュ化
     */
    public function setPasswordAttribute($value)
    {
        $this->attributes['password'] = bcrypt($value);
    }
    
    /**
     * メールアドレスを小文字に変換
     */
    public function setEmailAttribute($value)
    {
        $this->attributes['email'] = strtolower($value);
    }
    
    /*
    |--------------------------------------------------------------------------
    | スコープ
    |--------------------------------------------------------------------------
    */
    
    /**
     * アクティブなユーザーのみ
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
    
    /**
     * 管理者のみ
     */
    public function scopeAdmins($query)
    {
        return $query->where('is_admin', true);
    }
    
    /**
     * メール認証済みのみ
     */
    public function scopeVerified($query)
    {
        return $query->whereNotNull('email_verified_at');
    }
    
    /**
     * 最近ログインしたユーザー
     */
    public function scopeRecentlyActive($query, $days = 7)
    {
        return $query->where('last_login_at', '>=', now()->subDays($days));
    }
    
    /**
     * 検索スコープ
     */
    public function scopeSearch($query, $keyword)
    {
        return $query->where(function ($q) use ($keyword) {
            $q->where('name', 'like', "%{$keyword}%")
              ->orWhere('email', 'like', "%{$keyword}%")
              ->orWhere('bio', 'like', "%{$keyword}%");
        });
    }
    
    /*
    |--------------------------------------------------------------------------
    | ビジネスロジック
    |--------------------------------------------------------------------------
    */
    
    /**
     * ユーザーが管理者かチェック
     */
    public function isAdmin()
    {
        return $this->is_admin;
    }
    
    /**
     * ユーザーをフォロー
     */
    public function follow(User $user)
    {
        if ($this->id === $user->id) {
            throw new \Exception('自分自身をフォローすることはできません。');
        }
        
        $this->following()->syncWithoutDetaching($user->id);
    }
    
    /**
     * フォローを解除
     */
    public function unfollow(User $user)
    {
        $this->following()->detach($user->id);
    }
    
    /**
     * フォローしているかチェック
     */
    public function isFollowing(User $user)
    {
        return $this->following()->where('following_id', $user->id)->exists();
    }
    
    /**
     * 投稿にいいね
     */
    public function likePost(Post $post)
    {
        $this->likedPosts()->syncWithoutDetaching($post->id);
        $post->increment('like_count');
    }
    
    /**
     * いいねを取り消し
     */
    public function unlikePost(Post $post)
    {
        $this->likedPosts()->detach($post->id);
        $post->decrement('like_count');
    }
    
    /**
     * ログイン処理
     */
    public function recordLogin()
    {
        $this->increment('login_count');
        $this->update(['last_login_at' => now()]);
    }
    
    /**
     * 統計情報を取得
     */
    public function getStats()
    {
        return [
            'posts_count' => $this->posts()->count(),
            'comments_count' => $this->comments()->count(),
            'followers_count' => $this->followers()->count(),
            'following_count' => $this->following()->count(),
            'liked_posts_count' => $this->likedPosts()->count(),
        ];
    }
    
    /*
    |--------------------------------------------------------------------------
    | イベント
    |--------------------------------------------------------------------------
    */
    
    /**
     * モデルイベントの登録
     */
    protected static function booted()
    {
        // ユーザー削除時の処理
        static::deleting(function ($user) {
            // ソフトデリートの場合は処理しない
            if (!$user->isForceDeleting()) {
                return;
            }
            
            // アバター画像を削除
            if ($user->avatar) {
                \Storage::disk('public')->delete($user->avatar);
            }
            
            // 関連データを削除（カスケード削除が設定されていない場合）
            $user->posts()->delete();
            $user->comments()->delete();
        });
    }
}