# 第5章 サンプルコード - 実用的なCRUDアプリケーション開発

## 概要

第5章「実用的なCRUDアプリケーション開発」で使用するサンプルコード集です。
RESTful な CRUD 操作、フォーム処理とバリデーション、ページネーション・検索機能、Alpine.js による軽量インタラクションを含む、完全なタスク管理システムを実装しています。

## 📋 機能一覧

### ✨ 基本的なCRUD機能
- **タスクの作成・読取・更新・削除** - RESTful な7つのアクション
- **バリデーション機能** - Form Request による堅牢なデータ検証
- **ソフトデリート** - 論理削除によるデータ保護
- **一括操作** - 複数タスクの同時処理

### 🔍 検索・フィルタリング機能
- **複合検索** - タイトル、説明、担当者、タグによる横断検索
- **フィルタリング** - ステータス、優先度、カテゴリ、期限日による絞り込み
- **ページネーション** - 大量データの効率的な表示
- **ソート機能** - 作成日、期限日、優先度による並び替え

### 🎨 ユーザーインターフェース
- **レスポンシブデザイン** - Bootstrap 5 による美しい UI
- **Alpine.js** - 軽量でリアクティブなフロントエンド
- **表示切替** - カード表示・テーブル表示の選択可能
- **リアルタイム更新** - Ajax による非同期処理

### 🚀 高度な機能
- **添付ファイル** - ファイルアップロード・ダウンロード機能
- **タグ管理** - 柔軟なタスク分類システム
- **進捗管理** - 進捗率とステータス連動
- **統計情報** - ダッシュボードとレポート機能

## 📁 ファイル構成

```
chapter05/
├── controllers/                         # コントローラー
│   └── TaskController.php              # メインのタスク管理コントローラー
├── models/                              # モデル
│   └── Task.php                        # タスクモデル（Eloquent ORM）
├── requests/                            # フォームリクエスト
│   └── TaskRequest.php                 # タスクバリデーション
├── views/                              # ビューテンプレート
│   ├── layouts/
│   │   └── app.blade.php               # アプリケーションレイアウト
│   └── tasks/
│       ├── index.blade.php             # タスク一覧画面
│       └── create.blade.php            # タスク作成画面
├── routes/                             # ルーティング
│   └── web.php                         # Web ルート定義
├── migrations/                         # データベースマイグレーション
│   └── 2024_08_29_000001_create_tasks_table.php
├── factories/                          # ファクトリー（テストデータ生成）
├── seeders/                            # シーダー（初期データ投入）
├── config/                             # 設定ファイル
└── README.md                           # このファイル
```

## 🚀 セットアップ手順

### 1. 前提条件

- PHP 8.1 以降
- Laravel 10.x
- MySQL 8.0 または PostgreSQL 13 以降
- Node.js 16 以降（フロントエンドビルド用）
- Composer

### 2. ファイルの配置

#### コントローラーの配置
```bash
# TaskController を Laravel プロジェクトにコピー
cp controllers/TaskController.php app/Http/Controllers/
```

#### モデルの配置
```bash
# Task モデルを配置
cp models/Task.php app/Models/
```

#### リクエストクラスの配置
```bash
# フォームリクエストを配置
cp requests/TaskRequest.php app/Http/Requests/
```

#### ビューファイルの配置
```bash
# ビューファイルをコピー
cp -r views/* resources/views/
```

#### ルートファイルの配置
```bash
# ルート設定をマージ（既存のroutes/web.phpに追加）
# 既存ファイルのバックアップを推奨
cat routes/web.php >> routes/web.php
```

#### マイグレーションファイルの配置
```bash
# マイグレーションファイルをコピー
cp migrations/*.php database/migrations/
```

### 3. 依存関係モデルの準備

このサンプルコードは以下のモデルとの関連を前提としています：

#### User モデル（Laravel 標準）
```bash
# 既存のUserモデルを使用、必要に応じてリレーションを追加
```

#### Category モデル（第4章サンプルを流用可能）
```bash
# 第4章のCategoryモデルを使用、または以下で簡易版を作成
php artisan make:model Category -m
```

#### Tag モデル（多対多リレーション用）
```bash
# タグ機能用のモデル作成
php artisan make:model Tag -m
php artisan make:migration create_task_tag_table
```

### 4. データベース設定

#### 環境設定
```bash
# .env ファイルでデータベース接続を設定
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=laravel_chapter05
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

#### データベース作成
```bash
# MySQL の場合
mysql -u root -p
CREATE DATABASE laravel_chapter05 CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EXIT;
```

#### マイグレーション実行
```bash
# マイグレーション実行（テーブル作成）
php artisan migrate

# テストデータ投入（オプション）
php artisan db:seed
```

### 5. フロントエンド設定

#### NPM パッケージのインストール
```bash
# Bootstrap と Alpine.js （CDN使用のため基本不要）
# 必要に応じてカスタムビルドを作成
npm install
npm run dev
```

#### アセットコンパイル
```bash
# 本番環境向けビルド
npm run build
```

## 🔧 設定・カスタマイズ

### 1. モデル関連の設定

#### User モデルの拡張
```php
// app/Models/User.php に以下のリレーションを追加

public function tasks()
{
    return $this->hasMany(Task::class);
}

public function isAdmin(): bool
{
    return $this->role === 'admin'; // role カラムがある場合
}
```

#### Category モデルの作成
```php
// app/Models/Category.php
class Category extends Model
{
    protected $fillable = ['name', 'description', 'color', 'is_active'];
    
    public function tasks()
    {
        return $this->hasMany(Task::class);
    }
    
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
}
```

### 2. ルートの設定

#### 認証ミドルウェアの適用
```php
// routes/web.php で認証が必要なルートを保護
Route::middleware(['auth'])->group(function () {
    // タスク管理ルート
});
```

#### Policy による認可制御（オプション）
```bash
# TaskPolicy を作成
php artisan make:policy TaskPolicy --model=Task
```

### 3. 設定ファイルのカスタマイズ

#### ページネーション設定
```php
// config/app.php または AppServiceProvider
use Illuminate\Pagination\Paginator;

Paginator::useBootstrap();
```

#### ファイルアップロード設定
```php
// config/filesystems.php
'disks' => [
    'task_attachments' => [
        'driver' => 'local',
        'root' => storage_path('app/task-attachments'),
        'url' => env('APP_URL').'/storage/task-attachments',
        'visibility' => 'private',
    ],
],
```

## 💡 使用例とデモ

### 基本的な使用手順

#### 1. タスクの作成
1. ブラウザで `/tasks/create` にアクセス
2. 必要な情報を入力（タイトル、説明、カテゴリ、優先度など）
3. タグや添付ファイルを追加（オプション）
4. 「タスクを作成」ボタンをクリック

#### 2. タスクの検索・フィルタリング
1. タスク一覧画面で検索ボックスにキーワードを入力
2. ステータス、優先度、カテゴリでフィルタリング
3. 並び順を変更（作成日、期限日、優先度）
4. 表示形式を選択（カード・テーブル）

#### 3. 一括操作
1. 対象タスクにチェックを入れる
2. 一括操作ボタンから操作を選択
3. 「完了にする」または「削除」を実行

### デモデータの生成

#### ファクトリーの作成
```bash
# Task用のファクトリー作成
php artisan make:factory TaskFactory --model=Task
```

#### シーダーの実行
```bash
# テストデータを50件生成
php artisan tinker
>>> \App\Models\Task::factory()->count(50)->create(['user_id' => 1]);
```

## 🛠️ トラブルシューティング

### よくあるエラー

#### 1. Class 'App\Models\Task' not found
```bash
# オートロードの再生成
composer dump-autoload
```

#### 2. SQLSTATE[42S02]: Base table doesn't exist
```bash
# マイグレーション状況確認
php artisan migrate:status

# 未実行のマイグレーションを実行
php artisan migrate
```

#### 3. Route [tasks.index] not defined
```bash
# ルートキャッシュのクリア
php artisan route:clear
php artisan route:cache
```

#### 4. Alpine.js が動作しない
- ブラウザの開発者ツールでJavaScriptエラーを確認
- CDN の読み込み順序を確認
- `x-data` 属性の記述を確認

### デバッグ用コマンド

#### ルート一覧の確認
```bash
# 登録されているルートを確認
php artisan route:list --name=tasks
```

#### モデルのリレーション確認
```bash
php artisan tinker
>>> $task = \App\Models\Task::with(['user', 'category'])->first();
>>> $task->user->name;
>>> $task->category->name;
```

#### クエリログの確認
```php
// 生成されるSQLを確認
DB::enableQueryLog();
$tasks = Task::with('user', 'category')->paginate(15);
dd(DB::getQueryLog());
```

## 📚 学習のポイント

### 1. RESTful CRUD の理解
- 7つのアクション（index, create, store, show, edit, update, destroy）
- HTTP メソッドとURLの対応関係
- ルートモデルバインディングの活用

### 2. バリデーションとセキュリティ
- Form Request による集約的なバリデーション
- CSRF 保護の実装
- XSS 対策（Blade の自動エスケープ）
- SQLインジェクション対策（Eloquent ORM）

### 3. パフォーマンス最適化
- Eager Loading による N+1 問題の回避
- インデックスの適切な設計
- ページネーションによる大量データ対応
- キャッシュの活用

### 4. フロントエンド連携
- Alpine.js による軽量なリアクティブUI
- Ajax を使った非同期処理
- レスポンシブデザインの実装

## 🚀 拡張機能のアイデア

### 実装可能な追加機能

#### 1. 認証・認可の強化
- Laravel Breeze/Jetstream の導入
- Policy による細粒度な権限管理
- 2要素認証の実装

#### 2. 通知機能
- タスク期限のリマインダー
- ステータス変更通知
- メール・Slack通知

#### 3. API 開発
- RESTful API エンドポイントの作成
- JSON レスポンスの実装
- API認証（Sanctum）

#### 4. 高度なUI機能
- ドラッグ&ドロップによるステータス変更
- カンバンボード形式の表示
- カレンダービューの実装

## 📖 参考情報

### 公式ドキュメント
- [Laravel Controllers](https://laravel.com/docs/controllers)
- [Laravel Eloquent](https://laravel.com/docs/eloquent)
- [Laravel Validation](https://laravel.com/docs/validation)
- [Laravel Pagination](https://laravel.com/docs/pagination)

### フロントエンド関連
- [Alpine.js Documentation](https://alpinejs.dev/)
- [Bootstrap 5 Documentation](https://getbootstrap.com/docs/5.3/)
- [Font Awesome Icons](https://fontawesome.com/)

### 追加学習リソース
- [Laravel Best Practices](https://github.com/alexeymezenin/laravel-best-practices)
- [RESTful API Design](https://restfulapi.net/)
- [Database Design Best Practices](https://www.sisense.com/blog/better-sql-schema-design/)

---

## 📝 ライセンス・免責事項

このサンプルコードは教育目的で作成されています。本番環境での使用時は、セキュリティ対策やパフォーマンスチューニングを十分に検討してください。

**このサンプルコードを通じて、Laravelにおける実用的なCRUDアプリケーション開発スキルを身につけ、モダンなWebアプリケーション構築の基礎を習得することができます。**