<?php

/**
 * TaskController - RESTfulなタスク管理システム
 * 
 * 第5章「実用的なCRUDアプリケーション開発」サンプルコード
 * 
 * 機能:
 * - RESTful CRUD操作（7つのアクション）
 * - バリデーション機能付きフォーム処理
 * - ページネーションと検索機能
 * - パフォーマンス最適化（Eager Loading等）
 * - セキュリティ対策（CSRF、XSS、SQLインジェクション対策）
 */

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Task;
use App\Models\Category;
use App\Http\Requests\TaskRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;

class TaskController extends Controller
{
    /**
     * タスク一覧表示（検索・フィルタリング・ページネーション対応）
     * GET /tasks
     * 
     * @param Request $request 検索・フィルタ用パラメータ
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        // ベースクエリ（Eager Loadingでパフォーマンス最適化）
        $query = Task::select([
                    'id', 'title', 'description', 'status', 'priority',
                    'due_date', 'created_at', 'user_id', 'category_id'
                ])
                ->with([
                    'user:id,name',
                    'category:id,name,color'
                ]);

        // 検索フィルタ（複数カラム対応）
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function($q) use ($search) {
                $q->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'LIKE', "%{$search}%");
                  })
                  ->orWhereHas('tags', function($tagQuery) use ($search) {
                      $tagQuery->where('name', 'LIKE', "%{$search}%");
                  });
            });
        }

        // ステータスフィルタ
        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        // 優先度フィルタ
        if ($request->filled('priority')) {
            $query->where('priority', $request->input('priority'));
        }

        // カテゴリフィルタ
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->input('category_id'));
        }

        // 期限日フィルタ（範囲指定）
        if ($request->filled('due_date_from')) {
            $query->where('due_date', '>=', $request->input('due_date_from'));
        }
        if ($request->filled('due_date_to')) {
            $query->where('due_date', '<=', $request->input('due_date_to'));
        }

        // ソート処理（デフォルトは最新順）
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDir = $request->input('sort_dir', 'desc');
        $query->orderBy($sortBy, $sortDir);

        // ページネーション実行
        $tasks = $query->paginate(15)->withQueryString();

        // フィルタ用データを準備
        $statusOptions = Task::getStatusOptions();
        $priorityOptions = Task::getPriorityOptions();
        $categories = Category::active()->get();

        return view('tasks.index', compact(
            'tasks',
            'statusOptions',
            'priorityOptions',
            'categories'
        ));
    }

    /**
     * タスク作成フォーム表示
     * GET /tasks/create
     * 
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $categories = Category::active()
                        ->orderBy('name')
                        ->get();
        $priorityOptions = Task::getPriorityOptions();

        return view('tasks.create', compact('categories', 'priorityOptions'));
    }

    /**
     * タスク作成処理（バリデーション・セキュリティ対策済み）
     * POST /tasks
     * 
     * @param TaskRequest $request 自動バリデーション済みリクエスト
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(TaskRequest $request)
    {
        // TaskRequest により自動的にバリデーション実行
        // バリデーションエラー時は自動的にリダイレクト
        
        // バリデーション済みデータの取得
        $validated = $request->validated();
        
        // ファイルアップロード処理
        if ($request->hasFile('attachments')) {
            $validated['attachments'] = $this->handleFileUploads($request->file('attachments'));
        }
        
        // タスク作成（Mass Assignment Protection対応）
        $task = Task::create(array_merge($validated, [
            'user_id' => auth()->id(), // 現在ログイン中のユーザーIDを自動設定
        ]));
        
        // タグの関連付け（多対多リレーション）
        if ($request->has('tags')) {
            $this->attachTags($task, $request->input('tags'));
        }

        return redirect()
            ->route('tasks.show', $task)
            ->with('success', 'タスクを作成しました。');
    }

    /**
     * タスク詳細表示
     * GET /tasks/{id}
     * 
     * @param Task $task モデルバインディング
     * @return \Illuminate\View\View
     */
    public function show(Task $task)
    {
        // 関連データも一緒に取得（N+1問題対策）
        $task->load([
            'user',
            'category',
            'tags',
            'comments.user',
            'attachments'
        ]);

        return view('tasks.show', compact('task'));
    }

    /**
     * タスク編集フォーム表示
     * GET /tasks/{id}/edit
     * 
     * @param Task $task モデルバインディング
     * @return \Illuminate\View\View
     */
    public function edit(Task $task)
    {
        $categories = Category::active()
                        ->orderBy('name')
                        ->get();
        $priorityOptions = Task::getPriorityOptions();

        // 現在のタグIDを取得
        $selectedTags = $task->tags->pluck('id')->toArray();

        return view('tasks.edit', compact(
            'task',
            'categories',
            'priorityOptions',
            'selectedTags'
        ));
    }

    /**
     * タスク更新処理（認可制御・バリデーション対応）
     * PUT/PATCH /tasks/{id}
     * 
     * @param TaskRequest $request バリデーション済みリクエスト
     * @param Task $task モデルバインディング
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(TaskRequest $request, Task $task)
    {
        // 更新権限チェック（Policy使用）
        $this->authorize('update', $task);
        
        $validated = $request->validated();
        
        // ファイル処理
        if ($request->hasFile('attachments')) {
            $validated['attachments'] = $this->handleFileUploads($request->file('attachments'));
        }

        // タスク更新
        $task->update($validated);
        
        // タグの更新（既存の関連を削除してから新しい関連を作成）
        if ($request->has('tags')) {
            $this->syncTags($task, $request->input('tags'));
        }

        return redirect()
            ->route('tasks.show', $task)
            ->with('success', 'タスクを更新しました。');
    }

    /**
     * タスク削除処理（認可制御対応）
     * DELETE /tasks/{id}
     * 
     * @param Task $task モデルバインディング
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Task $task)
    {
        // 削除権限チェック
        $this->authorize('delete', $task);

        // 関連ファイルの削除
        if ($task->attachments) {
            foreach (json_decode($task->attachments, true) as $file) {
                Storage::disk('public')->delete($file['path']);
            }
        }

        // ソフトデリート実行
        $task->delete();

        return redirect()
            ->route('tasks.index')
            ->with('success', 'タスクを削除しました。');
    }

    /**
     * タスク統計情報表示（キャッシュ対応）
     * GET /tasks/statistics
     * 
     * @return \Illuminate\View\View
     */
    public function statistics()
    {
        // 統計データをキャッシュして高速化
        $stats = Cache::remember('task_statistics', 300, function() {
            return [
                'total' => Task::count(),
                'completed' => Task::completed()->count(),
                'overdue' => Task::overdue()->count(),
                'high_priority' => Task::where('priority', '>=', 3)->count(),
                'by_status' => Task::groupBy('status')
                                  ->selectRaw('status, count(*) as count')
                                  ->pluck('count', 'status'),
                'by_category' => Task::join('categories', 'tasks.category_id', '=', 'categories.id')
                                    ->groupBy('categories.name')
                                    ->selectRaw('categories.name, count(*) as count')
                                    ->pluck('count', 'name'),
            ];
        });

        return view('tasks.statistics', compact('stats'));
    }

    /**
     * 一括操作（複数タスクの状態変更）
     * POST /tasks/bulk-action
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'task_ids' => 'required|array',
            'task_ids.*' => 'exists:tasks,id',
            'action' => 'required|string|in:complete,delete,archive'
        ]);

        $taskIds = $request->input('task_ids');
        $action = $request->input('action');

        // ユーザーが所有するタスクのみに制限（セキュリティ対策）
        $tasks = Task::whereIn('id', $taskIds)
                    ->where('user_id', auth()->id())
                    ->get();

        $updated = 0;
        foreach ($tasks as $task) {
            switch ($action) {
                case 'complete':
                    $task->update(['status' => 'completed', 'completed_at' => now()]);
                    $updated++;
                    break;
                case 'delete':
                    $task->delete();
                    $updated++;
                    break;
                case 'archive':
                    $task->update(['status' => 'archived', 'archived_at' => now()]);
                    $updated++;
                    break;
            }
        }

        return response()->json([
            'success' => true,
            'message' => "{$updated}件のタスクを{$action}しました。",
            'updated_count' => $updated
        ]);
    }

    /**
     * ファイルアップロード処理
     * 
     * @param array $files アップロードファイル配列
     * @return string JSON形式のファイル情報
     */
    private function handleFileUploads($files)
    {
        $uploadedFiles = [];
        
        foreach ($files as $file) {
            // ファイルバリデーション
            if ($file->isValid()) {
                $path = $file->store('task-attachments', 'public');
                $uploadedFiles[] = [
                    'original_name' => $file->getClientOriginalName(),
                    'path' => $path,
                    'size' => $file->getSize(),
                    'mime_type' => $file->getMimeType(),
                ];
            }
        }
        
        return json_encode($uploadedFiles);
    }

    /**
     * タグの関連付け
     * 
     * @param Task $task
     * @param array $tagIds
     */
    private function attachTags(Task $task, array $tagIds)
    {
        $task->tags()->attach($tagIds);
    }

    /**
     * タグの同期（既存関連削除→新規関連作成）
     * 
     * @param Task $task
     * @param array $tagIds
     */
    private function syncTags(Task $task, array $tagIds)
    {
        $task->tags()->sync($tagIds);
    }
}