<?php

/**
 * TaskRequest - タスクのフォームリクエストクラス
 * 
 * 第5章「実用的なCRUDアプリケーション開発」サンプルコード
 * 
 * 機能:
 * - 入力データの自動バリデーション
 * - カスタムバリデーションルール
 * - エラーメッセージのカスタマイズ
 * - セキュリティ対策（XSS、データ検証）
 * - 条件付きバリデーション
 */

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class TaskRequest extends FormRequest
{
    /**
     * リクエストを実行する権限があるかを判定
     * 
     * @return bool
     */
    public function authorize(): bool
    {
        // 認証済みユーザーのみがタスクを作成・更新可能
        return auth()->check();
    }

    /**
     * バリデーションルールを定義
     * 
     * @return array<string, mixed>
     */
    public function rules(): array
    {
        // 基本ルール
        $rules = [
            'title' => ['required', 'string', 'max:255', 'min:3'],
            'description' => ['nullable', 'string', 'max:2000'],
            'category_id' => ['required', 'integer', 'exists:categories,id'],
            'priority' => ['required', 'integer', 'between:1,4'],
            'due_date' => ['nullable', 'date', 'after:today'],
            'status' => ['required', 'string', Rule::in(['pending', 'in_progress', 'completed', 'archived'])],
            'progress' => ['nullable', 'integer', 'between:0,100'],
            'estimated_hours' => ['nullable', 'numeric', 'min:0.1', 'max:999.99'],
            'tags' => ['nullable', 'array', 'max:10'],
            'tags.*' => ['string', 'max:50', 'regex:/^[a-zA-Z0-9\p{L}\s\-_]+$/u'],
            'attachments' => ['nullable', 'array', 'max:5'],
            'attachments.*' => [
                'file',
                'max:10240', // 10MB
                'mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,gif,txt,csv'
            ],
        ];

        // 更新時の追加ルール
        if ($this->isMethod('PUT') || $this->isMethod('PATCH')) {
            // 更新時は自分のタスクのみ編集可能
            $rules['id'] = ['required', 'integer', 'exists:tasks,id'];
        }

        // ステータスが完了の場合の条件付きバリデーション
        if ($this->input('status') === 'completed') {
            $rules['completed_at'] = ['nullable', 'date', 'before_or_equal:now'];
            $rules['completion_note'] = ['nullable', 'string', 'max:500'];
            $rules['progress'] = ['required', 'integer', 'in:100']; // 完了時は100%必須
        }

        // 優先度が高い場合の追加要件
        if ($this->input('priority') >= 3) {
            $rules['reason'] = ['required', 'string', 'max:300', 'min:10'];
        }

        // 期限日が設定されている場合の追加バリデーション
        if ($this->filled('due_date')) {
            $rules['reminder_days'] = ['nullable', 'integer', 'min:1', 'max:30'];
        }

        return $rules;
    }

    /**
     * カスタムエラーメッセージを定義
     * 
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            // タイトル関連
            'title.required' => 'タスクのタイトルは必須入力です。',
            'title.string' => 'タイトルは文字列で入力してください。',
            'title.max' => 'タイトルは255文字以内で入力してください。',
            'title.min' => 'タイトルは3文字以上で入力してください。',

            // 説明関連
            'description.string' => '説明は文字列で入力してください。',
            'description.max' => '説明は2000文字以内で入力してください。',

            // カテゴリ関連
            'category_id.required' => 'カテゴリの選択は必須です。',
            'category_id.integer' => 'カテゴリは正しい形式で選択してください。',
            'category_id.exists' => '選択されたカテゴリが存在しません。',

            // 優先度関連
            'priority.required' => '優先度の選択は必須です。',
            'priority.integer' => '優先度は数値で選択してください。',
            'priority.between' => '優先度は1（低）から4（緊急）の範囲で選択してください。',

            // 期限日関連
            'due_date.date' => '期限日は正しい日付形式で入力してください。',
            'due_date.after' => '期限日は今日以降の日付を設定してください。',

            // ステータス関連
            'status.required' => 'ステータスの選択は必須です。',
            'status.in' => '無効なステータスが選択されました。',

            // 進捗関連
            'progress.integer' => '進捗は数値で入力してください。',
            'progress.between' => '進捗は0%から100%の範囲で入力してください。',
            'progress.in' => '完了ステータスの場合、進捗は100%である必要があります。',

            // 工数関連
            'estimated_hours.numeric' => '予定工数は数値で入力してください。',
            'estimated_hours.min' => '予定工数は0.1時間以上で入力してください。',
            'estimated_hours.max' => '予定工数は999.99時間以下で入力してください。',

            // タグ関連
            'tags.array' => 'タグは配列形式で送信してください。',
            'tags.max' => 'タグは最大10個まで設定可能です。',
            'tags.*.string' => 'タグは文字列で入力してください。',
            'tags.*.max' => '各タグは50文字以内で入力してください。',
            'tags.*.regex' => 'タグには英数字、ひらがな、カタカナ、漢字、スペース、ハイフン、アンダースコアのみ使用可能です。',

            // 添付ファイル関連
            'attachments.array' => '添付ファイルは配列形式で送信してください。',
            'attachments.max' => '添付ファイルは最大5個まで設定可能です。',
            'attachments.*.file' => '正しいファイルを選択してください。',
            'attachments.*.max' => '各ファイルのサイズは10MB以下にしてください。',
            'attachments.*.mimes' => '許可されているファイル形式: PDF, Word, Excel, 画像ファイル(JPG/PNG/GIF), テキストファイル(TXT/CSV)',

            // 条件付きバリデーション関連
            'completed_at.date' => '完了日時は正しい日付形式で入力してください。',
            'completed_at.before_or_equal' => '完了日時は現在時刻以前で設定してください。',
            'completion_note.string' => '完了メモは文字列で入力してください。',
            'completion_note.max' => '完了メモは500文字以内で入力してください。',
            'reason.required' => '高優先度のタスクには理由の記載が必要です。',
            'reason.string' => '理由は文字列で入力してください。',
            'reason.max' => '理由は300文字以内で入力してください。',
            'reason.min' => '理由は10文字以上で入力してください。',
            'reminder_days.integer' => 'リマインダー日数は数値で入力してください。',
            'reminder_days.min' => 'リマインダーは1日以上で設定してください。',
            'reminder_days.max' => 'リマインダーは30日以下で設定してください。',
        ];
    }

    /**
     * バリデーション用属性名をカスタマイズ
     * 
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'title' => 'タイトル',
            'description' => '説明',
            'category_id' => 'カテゴリ',
            'priority' => '優先度',
            'due_date' => '期限日',
            'status' => 'ステータス',
            'progress' => '進捗率',
            'estimated_hours' => '予定工数',
            'tags' => 'タグ',
            'attachments' => '添付ファイル',
            'completed_at' => '完了日時',
            'completion_note' => '完了メモ',
            'reason' => '理由',
            'reminder_days' => 'リマインダー日数',
        ];
    }

    /**
     * バリデーション前のデータ準備処理
     * 
     * @return void
     */
    protected function prepareForValidation(): void
    {
        // データの前処理
        $this->merge([
            'title' => $this->title ? trim($this->title) : null,
            'description' => $this->description ? trim($this->description) : null,
            'due_date' => $this->due_date ?: null, // 空文字をnullに変換
            'estimated_hours' => $this->estimated_hours ?: null,
            'progress' => $this->progress ?? 0, // デフォルト値を設定
        ]);

        // タグの前処理（空要素を除去）
        if ($this->has('tags') && is_array($this->tags)) {
            $this->merge([
                'tags' => array_filter(array_map('trim', $this->tags), function($tag) {
                    return !empty($tag);
                })
            ]);
        }
    }

    /**
     * バリデーター拡張（条件付きバリデーション）
     * 
     * @param \Illuminate\Validation\Validator $validator
     * @return void
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // カスタムバリデーション: 緊急優先度の場合は期限日必須
            if ($this->priority == 4 && !$this->due_date) {
                $validator->errors()->add('due_date', '緊急優先度のタスクには期限日の設定が必要です。');
            }

            // カスタムバリデーション: 完了ステータスなのに期限日が未来日
            if ($this->status === 'completed' && $this->due_date && $this->due_date > now()->toDateString()) {
                $validator->errors()->add('due_date', '完了済みタスクの期限日は過去または今日の日付である必要があります。');
            }

            // カスタムバリデーション: 進捗率とステータスの整合性チェック
            if ($this->progress == 100 && $this->status !== 'completed') {
                $validator->errors()->add('status', '進捗率が100%の場合、ステータスは「完了」である必要があります。');
            }

            // カスタムバリデーション: ファイルサイズの合計チェック
            if ($this->hasFile('attachments')) {
                $totalSize = 0;
                foreach ($this->file('attachments') as $file) {
                    $totalSize += $file->getSize();
                }
                // 50MB制限
                if ($totalSize > 50 * 1024 * 1024) {
                    $validator->errors()->add('attachments', '添付ファイルの合計サイズは50MB以下である必要があります。');
                }
            }

            // カスタムバリデーション: タグの重複チェック
            if ($this->has('tags') && is_array($this->tags)) {
                $uniqueTags = array_unique(array_map('strtolower', $this->tags));
                if (count($this->tags) !== count($uniqueTags)) {
                    $validator->errors()->add('tags', '重複するタグが含まれています。');
                }
            }
        });

        // 条件付きルール: チームタスクの場合はチームID必須
        $validator->sometimes('team_id', 'required|exists:teams,id', function ($input) {
            return $input->is_team_task === true;
        });

        // 条件付きルール: 緊急タスクは添付ファイル必須
        $validator->sometimes('attachments', 'required', function ($input) {
            return $input->priority === 4;
        });

        // 条件付きルール: 承認が必要なタスクの場合
        $validator->sometimes('approver_id', 'required|exists:users,id', function ($input) {
            return $input->requires_approval === true;
        });
    }

    /**
     * バリデーション済みデータを取得（セキュリティ対策付き）
     * 
     * @param array|null $key
     * @param mixed $default
     * @return array
     */
    public function validated($key = null, $default = null): array
    {
        $validated = parent::validated($key, $default);
        
        // XSS対策: HTMLタグを除去
        if (isset($validated['title'])) {
            $validated['title'] = strip_tags($validated['title']);
        }
        
        if (isset($validated['description'])) {
            $validated['description'] = strip_tags($validated['description'], '<p><br><strong><em><ul><ol><li>');
        }

        // セキュリティ: 現在ユーザーのIDを自動設定
        if (!isset($validated['user_id'])) {
            $validated['user_id'] = auth()->id();
        }

        return $validated;
    }

    /**
     * フィルタ済みデータを取得（不要フィールドを除去）
     * 
     * @return array
     */
    public function getSafeData(): array
    {
        $validated = $this->validated();
        
        // 内部処理用フィールドを除去
        unset($validated['_token'], $validated['_method']);
        
        return $validated;
    }
}