@extends('layouts.app')

@section('title', 'タスク作成')

@section('header')
    <div class="d-flex align-items-center">
        <a href="{{ route('tasks.index') }}" class="btn btn-outline-secondary me-3">
            <i class="fas fa-arrow-left me-1"></i>一覧に戻る
        </a>
        <div>
            <h2 class="mb-0">
                <i class="fas fa-plus text-primary me-2"></i>新規タスク作成
            </h2>
            <p class="text-muted mb-0">新しいタスクを作成します</p>
        </div>
    </div>
@endsection

@section('content')
<div class="row justify-content-center">
    <div class="col-lg-8">
        <form method="POST" action="{{ route('tasks.store') }}" 
              enctype="multipart/form-data" 
              novalidate
              x-data="taskForm()">
            @csrf
            
            <!-- 基本情報カード -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-info-circle me-2"></i>基本情報</h5>
                </div>
                <div class="card-body">
                    <!-- タイトル -->
                    <div class="mb-3">
                        <label for="title" class="form-label">
                            タスクタイトル <span class="text-danger">*</span>
                        </label>
                        <input type="text" 
                               class="form-control @error('title') is-invalid @enderror"
                               id="title" 
                               name="title" 
                               value="{{ old('title') }}"
                               placeholder="例：Laravel学習の進捗確認"
                               x-model="form.title"
                               required>
                        @error('title')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @else
                            <div class="form-text">3文字以上255文字以内で入力してください</div>
                        @enderror
                    </div>

                    <!-- 説明文 -->
                    <div class="mb-3">
                        <label for="description" class="form-label">説明</label>
                        <textarea class="form-control @error('description') is-invalid @enderror"
                                  id="description" 
                                  name="description" 
                                  rows="4"
                                  placeholder="タスクの詳細な説明を入力してください（任意）"
                                  x-model="form.description">{{ old('description') }}</textarea>
                        @error('description')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @else
                            <div class="form-text">
                                <span x-text="form.description ? form.description.length : 0"></span>/2000文字
                            </div>
                        @enderror
                    </div>

                    <div class="row">
                        <!-- カテゴリ -->
                        <div class="col-md-6">
                            <label for="category_id" class="form-label">
                                カテゴリ <span class="text-danger">*</span>
                            </label>
                            <select class="form-select @error('category_id') is-invalid @enderror"
                                    id="category_id" 
                                    name="category_id" 
                                    x-model="form.category_id"
                                    required>
                                <option value="">カテゴリを選択してください</option>
                                @foreach($categories as $category)
                                    <option value="{{ $category->id }}" 
                                            {{ old('category_id') == $category->id ? 'selected' : '' }}>
                                        {{ $category->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('category_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- ステータス -->
                        <div class="col-md-6">
                            <label for="status" class="form-label">
                                ステータス <span class="text-danger">*</span>
                            </label>
                            <select class="form-select @error('status') is-invalid @enderror"
                                    id="status" 
                                    name="status" 
                                    x-model="form.status"
                                    required>
                                <option value="pending" {{ old('status', 'pending') == 'pending' ? 'selected' : '' }}>
                                    未着手
                                </option>
                                <option value="in_progress" {{ old('status') == 'in_progress' ? 'selected' : '' }}>
                                    進行中
                                </option>
                                <option value="completed" {{ old('status') == 'completed' ? 'selected' : '' }}>
                                    完了
                                </option>
                            </select>
                            @error('status')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- 詳細設定カード -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-cog me-2"></i>詳細設定</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <!-- 優先度 -->
                        <div class="col-md-6">
                            <label for="priority" class="form-label">
                                優先度 <span class="text-danger">*</span>
                            </label>
                            <select class="form-select @error('priority') is-invalid @enderror"
                                    id="priority" 
                                    name="priority" 
                                    x-model="form.priority"
                                    @change="checkPriorityRequirements()"
                                    required>
                                @foreach($priorityOptions as $value => $label)
                                    <option value="{{ $value }}" 
                                            {{ old('priority', 2) == $value ? 'selected' : '' }}>
                                        {{ $label }}
                                    </option>
                                @endforeach
                            </select>
                            @error('priority')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- 進捗率 -->
                        <div class="col-md-6">
                            <label for="progress" class="form-label">進捗率</label>
                            <div class="input-group">
                                <input type="range" 
                                       class="form-range me-3" 
                                       id="progress" 
                                       name="progress" 
                                       min="0" 
                                       max="100" 
                                       value="{{ old('progress', 0) }}"
                                       x-model.number="form.progress"
                                       @input="updateProgressDisplay()">
                                <span class="input-group-text" x-text="form.progress + '%'">0%</span>
                            </div>
                            @error('progress')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="row mt-3">
                        <!-- 期限日 -->
                        <div class="col-md-6">
                            <label for="due_date" class="form-label">
                                期限日
                                <span x-show="form.priority >= 3" class="text-danger">*</span>
                            </label>
                            <input type="date" 
                                   class="form-control @error('due_date') is-invalid @enderror"
                                   id="due_date" 
                                   name="due_date" 
                                   value="{{ old('due_date') }}"
                                   min="{{ now()->toDateString() }}"
                                   x-model="form.due_date"
                                   :required="form.priority >= 3">
                            @error('due_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @else
                                <div class="form-text" x-show="form.priority >= 3">
                                    高優先度・緊急のタスクには期限日が必要です
                                </div>
                            @enderror
                        </div>

                        <!-- 予定工数 -->
                        <div class="col-md-6">
                            <label for="estimated_hours" class="form-label">予定工数（時間）</label>
                            <input type="number" 
                                   class="form-control @error('estimated_hours') is-invalid @enderror"
                                   id="estimated_hours" 
                                   name="estimated_hours" 
                                   value="{{ old('estimated_hours') }}"
                                   step="0.5" 
                                   min="0.1" 
                                   max="999.99"
                                   placeholder="例：8.5"
                                   x-model="form.estimated_hours">
                            @error('estimated_hours')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @else
                                <div class="form-text">作業にかかる予定時間を入力してください（任意）</div>
                            @enderror
                        </div>
                    </div>

                    <!-- 高優先度の理由 -->
                    <div x-show="form.priority >= 3" x-transition class="mt-3">
                        <label for="reason" class="form-label">
                            高優先度の理由 <span class="text-danger">*</span>
                        </label>
                        <textarea class="form-control @error('reason') is-invalid @enderror"
                                  id="reason" 
                                  name="reason" 
                                  rows="3"
                                  placeholder="なぜこのタスクが高優先度なのか、理由を明確に記載してください"
                                  x-model="form.reason"
                                  :required="form.priority >= 3">{{ old('reason') }}</textarea>
                        @error('reason')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @else
                            <div class="form-text">10文字以上300文字以内で入力してください</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- タグ・添付ファイルカード -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-paperclip me-2"></i>タグ・添付ファイル</h5>
                </div>
                <div class="card-body">
                    <!-- タグ -->
                    <div class="mb-3">
                        <label for="tags" class="form-label">タグ</label>
                        <div x-data="tagInput()">
                            <div class="input-group mb-2">
                                <input type="text" 
                                       class="form-control" 
                                       x-model="newTag"
                                       @keydown.enter.prevent="addTag()"
                                       placeholder="タグを入力してEnterキー">
                                <button type="button" 
                                        class="btn btn-outline-secondary" 
                                        @click="addTag()">
                                    追加
                                </button>
                            </div>
                            
                            <!-- 選択済みタグ表示 -->
                            <div x-show="tags.length > 0" class="mb-2">
                                <template x-for="(tag, index) in tags" :key="index">
                                    <span class="badge bg-primary me-2 mb-1">
                                        <span x-text="tag"></span>
                                        <button type="button" 
                                                class="btn-close btn-close-white ms-2" 
                                                style="font-size: 0.7em;"
                                                @click="removeTag(index)">
                                        </button>
                                    </span>
                                </template>
                            </div>
                            
                            <!-- Hidden inputs for tags -->
                            <template x-for="(tag, index) in tags" :key="index">
                                <input type="hidden" name="tags[]" :value="tag">
                            </template>
                            
                            @error('tags')
                                <div class="text-danger small">{{ $message }}</div>
                            @else
                                <div class="form-text">
                                    最大10個まで設定可能です。検索やフィルタリングに使用されます。
                                </div>
                            @enderror
                        </div>
                    </div>

                    <!-- 添付ファイル -->
                    <div class="mb-3">
                        <label for="attachments" class="form-label">
                            添付ファイル
                            <span x-show="form.priority == 4" class="text-danger">*</span>
                        </label>
                        <input type="file" 
                               class="form-control @error('attachments') @error('attachments.*') is-invalid @enderror @enderror"
                               id="attachments" 
                               name="attachments[]" 
                               multiple
                               accept=".pdf,.doc,.docx,.xls,.xlsx,.jpg,.jpeg,.png,.gif,.txt,.csv"
                               :required="form.priority == 4"
                               @change="handleFileSelection($event)">
                        
                        <!-- ファイル情報表示 -->
                        <div x-show="selectedFiles.length > 0" class="mt-2">
                            <div class="small text-muted mb-2">選択されたファイル:</div>
                            <template x-for="file in selectedFiles" :key="file.name">
                                <div class="d-flex justify-content-between align-items-center p-2 border rounded mb-1">
                                    <div>
                                        <i class="fas fa-file me-2"></i>
                                        <span x-text="file.name"></span>
                                        <small class="text-muted ms-2" x-text="formatFileSize(file.size)"></small>
                                    </div>
                                </div>
                            </template>
                        </div>
                        
                        @error('attachments')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        @error('attachments.*')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text" x-show="form.priority == 4">
                            緊急タスクには添付ファイルが必要です
                        </div>
                        <div class="form-text" x-show="form.priority != 4">
                            PDF, Word, Excel, 画像ファイル等。各ファイル10MB以下、最大5個まで
                        </div>
                    </div>
                </div>
            </div>

            <!-- 完了関連情報（完了ステータスの場合のみ） -->
            <div x-show="form.status === 'completed'" x-transition class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-check-circle me-2"></i>完了情報</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <!-- 完了日時 -->
                        <div class="col-md-6">
                            <label for="completed_at" class="form-label">
                                完了日時 <span class="text-danger">*</span>
                            </label>
                            <input type="datetime-local" 
                                   class="form-control @error('completed_at') is-invalid @enderror"
                                   id="completed_at" 
                                   name="completed_at" 
                                   value="{{ old('completed_at') }}"
                                   max="{{ now()->format('Y-m-d\TH:i') }}"
                                   x-model="form.completed_at"
                                   :required="form.status === 'completed'">
                            @error('completed_at')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <!-- 完了メモ -->
                    <div class="mt-3">
                        <label for="completion_note" class="form-label">
                            完了メモ <span class="text-danger">*</span>
                        </label>
                        <textarea class="form-control @error('completion_note') is-invalid @enderror"
                                  id="completion_note" 
                                  name="completion_note" 
                                  rows="3"
                                  placeholder="完了時の詳細、成果物、次回への申し送りなどを記載してください"
                                  x-model="form.completion_note"
                                  :required="form.status === 'completed'">{{ old('completion_note') }}</textarea>
                        @error('completion_note')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @else
                            <div class="form-text">完了状態の場合は入力必須です</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- 送信ボタン -->
            <div class="card">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <a href="{{ route('tasks.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-1"></i>キャンセル
                        </a>
                        
                        <div class="d-flex gap-2">
                            <button type="button" 
                                    class="btn btn-outline-primary"
                                    @click="saveDraft()">
                                <i class="fas fa-save me-1"></i>下書き保存
                            </button>
                            <button type="submit" 
                                    class="btn btn-primary"
                                    :disabled="!isFormValid()"
                                    :class="{ 'btn-primary': isFormValid(), 'btn-secondary': !isFormValid() }">
                                <i class="fas fa-check me-1"></i>タスクを作成
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>
@endsection

@push('scripts')
<script>
function taskForm() {
    return {
        form: {
            title: '{{ old("title") }}',
            description: '{{ old("description") }}',
            category_id: '{{ old("category_id") }}',
            status: '{{ old("status", "pending") }}',
            priority: {{ old('priority', 2) }},
            progress: {{ old('progress', 0) }},
            due_date: '{{ old("due_date") }}',
            estimated_hours: '{{ old("estimated_hours") }}',
            reason: '{{ old("reason") }}',
            completed_at: '{{ old("completed_at") }}',
            completion_note: '{{ old("completion_note") }}'
        },
        selectedFiles: [],

        checkPriorityRequirements() {
            // 優先度が高い場合は期限日を必須にする
            if (this.form.priority >= 3 && !this.form.due_date) {
                this.form.due_date = '';
            }
        },

        updateProgressDisplay() {
            // 進捗率が100%になった場合、ステータスを完了に変更するか確認
            if (this.form.progress === 100 && this.form.status !== 'completed') {
                if (confirm('進捗率が100%です。ステータスを「完了」に変更しますか？')) {
                    this.form.status = 'completed';
                    this.form.completed_at = new Date().toISOString().slice(0, 16);
                }
            }
        },

        handleFileSelection(event) {
            this.selectedFiles = Array.from(event.target.files);
            
            // ファイルサイズチェック
            let totalSize = 0;
            for (let file of this.selectedFiles) {
                totalSize += file.size;
                if (file.size > 10 * 1024 * 1024) { // 10MB
                    alert(`${file.name} のサイズが10MBを超えています。`);
                    event.target.value = '';
                    this.selectedFiles = [];
                    return;
                }
            }
            
            if (totalSize > 50 * 1024 * 1024) { // 合計50MB
                alert('添付ファイルの合計サイズが50MBを超えています。');
                event.target.value = '';
                this.selectedFiles = [];
            }
        },

        formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        },

        isFormValid() {
            const required = this.form.title && this.form.category_id && this.form.status && this.form.priority;
            
            if (!required) return false;
            
            // 高優先度の場合の追加バリデーション
            if (this.form.priority >= 3) {
                if (!this.form.due_date || !this.form.reason) return false;
            }
            
            // 完了ステータスの場合の追加バリデーション
            if (this.form.status === 'completed') {
                if (!this.form.completed_at || !this.form.completion_note) return false;
            }
            
            // 緊急タスクの場合は添付ファイル必須
            if (this.form.priority == 4 && this.selectedFiles.length === 0) return false;
            
            return true;
        },

        async saveDraft() {
            // 下書き保存の実装（オプション機能）
            alert('下書き保存機能はデモ版では利用できません。');
        }
    };
}

function tagInput() {
    return {
        newTag: '',
        tags: @json(old('tags', [])),

        addTag() {
            if (!this.newTag.trim()) return;
            
            if (this.tags.length >= 10) {
                alert('タグは最大10個まで設定可能です。');
                return;
            }
            
            const tag = this.newTag.trim();
            if (this.tags.includes(tag)) {
                alert('同じタグが既に追加されています。');
                return;
            }
            
            this.tags.push(tag);
            this.newTag = '';
        },

        removeTag(index) {
            this.tags.splice(index, 1);
        }
    };
}
</script>
@endpush