# 第6章 サンプルコード - 認証とセキュリティ

## 概要

第6章「認証とセキュリティ」で使用するサンプルコード集です。
Laravel Breezeを使用した安全な認証システムの実装から、ユーザー固有データの管理、包括的なセキュリティ対策まで、実用的なWebアプリケーションのセキュリティ機能を学習します。

## ファイル構成

```
chapter06/
├── controllers/                         # 認証・セキュリティ関連コントローラー
│   ├── TaskController.php              # ユーザー固有データ制御版タスクコントローラー
│   ├── AuthController.php              # カスタム認証コントローラー
│   └── SecurityController.php          # セキュリティ機能コントローラー
├── requests/                            # フォームリクエスト（バリデーション）
│   ├── StoreTaskRequest.php           # タスク作成リクエスト（XSS対策含む）
│   ├── UpdateTaskRequest.php          # タスク更新リクエスト
│   └── ProfileUpdateRequest.php       # プロフィール更新リクエスト
├── policies/                           # 認可制御ポリシー
│   ├── TaskPolicy.php                 # タスク認可ポリシー
│   └── UserPolicy.php                 # ユーザー認可ポリシー
├── middleware/                         # カスタムミドルウェア
│   ├── SecurityHeaders.php            # セキュリティヘッダー追加
│   ├── UserActivity.php               # ユーザー活動記録
│   └── AdminOnly.php                  # 管理者限定アクセス
├── migrations/                         # 認証・セキュリティ関連マイグレーション
│   ├── 2024_08_06_000000_add_user_fields_to_users_table.php
│   ├── 2024_08_06_100000_add_user_ownership_to_tasks.php
│   ├── 2024_08_06_200000_create_user_activities_table.php
│   └── 2024_08_06_300000_create_login_histories_table.php
├── views/                              # 認証・セキュリティ関連ビュー
│   ├── auth/                           # 認証関連ビュー
│   │   ├── login.blade.php            # ログインフォーム
│   │   ├── register.blade.php         # ユーザー登録フォーム
│   │   └── dashboard.blade.php        # ユーザーダッシュボード
│   ├── profile/                        # プロフィール管理
│   │   └── edit.blade.php             # プロフィール編集
│   └── admin/                          # 管理者専用画面
│       └── security-report.blade.php  # セキュリティレポート
├── tests/                              # セキュリティ関連テスト
│   ├── AuthenticationTest.php         # 認証機能テスト
│   ├── AuthorizationTest.php          # 認可制御テスト
│   └── SecurityTest.php               # セキュリティ機能テスト
├── config/                             # セキュリティ設定
│   ├── security.php                   # カスタムセキュリティ設定
│   └── session-custom.php             # セッション設定例
├── raw-php-examples/                   # 生PHP比較用サンプル
│   ├── vulnerable-login.php           # 脆弱性のある生PHPログイン
│   ├── insecure-session.php           # 不安全なセッション管理
│   └── sql-injection-demo.php         # SQLインジェクション例
└── README.md                          # このファイル
```

## セットアップ手順

### 1. Laravel Breezeのインストール

```bash
# Laravel Breezeパッケージをインストール
composer require laravel/breeze --dev

# Breezeの初期セットアップ（Bladeスタック）
php artisan breeze:install blade

# NPMパッケージのインストールとアセットビルド
npm install && npm run dev

# マイグレーションを実行
php artisan migrate
```

### 2. ファイルの配置

#### コントローラーファイルの配置
```bash
# コントローラーをコピー
cp controllers/*.php app/Http/Controllers/

# 既存のTaskControllerを上書き（バックアップ推奨）
cp controllers/TaskController.php app/Http/Controllers/TaskController.php
```

#### リクエストファイルの配置
```bash
# フォームリクエストをコピー
cp requests/*.php app/Http/Requests/
```

#### ポリシーファイルの配置
```bash
# ポリシーをコピー
cp policies/*.php app/Policies/
```

#### ミドルウェアの配置
```bash
# カスタムミドルウェアをコピー
cp middleware/*.php app/Http/Middleware/
```

#### マイグレーションファイルの配置
```bash
# マイグレーションファイルをコピー
cp migrations/*.php database/migrations/
```

#### ビューファイルの配置
```bash
# ビューファイルをコピー
cp -r views/* resources/views/
```

### 3. 設定ファイルの更新

#### ミドルウェアの登録
```bash
# app/Http/Kernel.php に追加設定が必要
# 詳細は各ミドルウェアファイルのコメントを参照
```

#### ポリシーの登録
```bash
# app/Providers/AuthServiceProvider.php に追加設定が必要
# 詳細はポリシーファイルのコメントを参照
```

### 4. データベースマイグレーション

```bash
# 新しいマイグレーションを実行
php artisan migrate

# 必要に応じてシーダーを実行
php artisan db:seed
```

## 実装内容詳細

### Laravel Breeze認証機能

#### 提供される基本機能
- **ユーザー登録**: 安全なバリデーション・パスワードハッシュ化
- **ログイン**: レート制限・セッション管理・Remember Me
- **ログアウト**: セッション無効化・CSRFトークン再生成
- **パスワードリセット**: メール認証・安全なトークン管理
- **メール認証**: 確認メール送信・認証状態管理
- **プロフィール管理**: 安全な情報更新・パスワード変更

#### セキュリティ特徴
```php
// 自動CSRF保護
@csrf

// レート制限（ブルートフォース攻撃対策）
$request->ensureIsNotRateLimited();

// 強力なパスワードハッシュ化
Hash::make($password);

// セッションセキュリティ
$request->session()->regenerate();
```

### ユーザー固有データ管理

#### 実装パターン
```php
// ユーザー所有権の自動制御
$tasks = Auth::user()->tasks()->paginate(15);

// Policy による認可制御
$this->authorize('view', $task);

// 自動ユーザーID設定
Auth::user()->tasks()->create($validatedData);
```

#### データ分離の仕組み
- **自動フィルタリング**: 現在ユーザーのデータのみ表示
- **所有権チェック**: 他ユーザーデータへの不正アクセス防止
- **Policy 認可**: 細粒度のアクセス制御
- **共有機能**: 明示的な共有設定による安全なデータ共有

### セキュリティ対策実装

#### CSRF（Cross-Site Request Forgery）対策
```blade
<!-- フォームでの自動CSRF保護 -->
<form method="POST" action="{{ route('tasks.store') }}">
    @csrf
    <!-- フォーム要素 -->
</form>
```

```javascript
// Ajax でのCSRF保護
headers: {
    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
}
```

#### XSS（Cross-Site Scripting）対策
```blade
<!-- 自動エスケープ（推奨） -->
{{ $task->title }}

<!-- 非エスケープ（信頼できるデータのみ） -->
{!! $task->safe_html_content !!}
```

#### SQLインジェクション対策
```php
// Eloquent ORM（自動保護）
Task::where('title', 'LIKE', '%' . $search . '%')->get();

// パラメータバインディング
DB::select('SELECT * FROM tasks WHERE user_id = ?', [Auth::id()]);
```

#### セッションセキュリティ
```php
// セッション設定の最適化
'secure' => true,        // HTTPS必須
'http_only' => true,     // XSS攻撃対策
'same_site' => 'lax',    // CSRF攻撃対策
'encrypt' => true,       // セッション暗号化
```

#### レート制限
```php
// ルート単位での制限
Route::middleware(['throttle:60,1'])->group(function () {
    // 通常のページアクセス
});

Route::middleware(['throttle:30,1'])->group(function () {
    // API エンドポイント
});
```

#### セキュリティヘッダー
```php
// セキュリティヘッダーの自動追加
'X-Content-Type-Options' => 'nosniff',
'X-Frame-Options' => 'DENY',
'X-XSS-Protection' => '1; mode=block',
'Content-Security-Policy' => $cspPolicy,
```

## 使用例とパターン

### 基本的な認証フロー

#### ユーザー登録
```php
// 1. バリデーション（自動実行）
$request->validate([
    'name' => ['required', 'string', 'max:255'],
    'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users'],
    'password' => ['required', 'confirmed', Rules\Password::defaults()],
]);

// 2. 安全なユーザー作成
$user = User::create([
    'name' => $request->name,
    'email' => $request->email,
    'password' => Hash::make($request->password),
]);

// 3. イベント発火・自動ログイン
event(new Registered($user));
Auth::login($user);
```

#### ログイン処理
```php
// 1. レート制限チェック
$request->ensureIsNotRateLimited();

// 2. 認証実行
Auth::attempt($credentials, $request->boolean('remember'));

// 3. セキュリティ強化
$request->session()->regenerate();
```

### ユーザー固有データの操作

#### 一覧表示（フィルタリング）
```php
public function index(Request $request)
{
    $query = Auth::user()->tasks()->with('category');
    
    // 検索フィルタ（ユーザー固有データ内で検索）
    if ($request->filled('search')) {
        $query->where('title', 'LIKE', '%' . $request->search . '%');
    }
    
    $tasks = $query->latest()->paginate(15);
    
    return view('tasks.index', compact('tasks'));
}
```

#### データ作成（所有権自動設定）
```php
public function store(StoreTaskRequest $request)
{
    // 現在のユーザーのタスクとして自動作成
    $task = Auth::user()->tasks()->create($request->validated());
    
    return redirect()->route('tasks.show', $task)
                    ->with('success', 'タスクが作成されました。');
}
```

#### 認可制御（Policy使用）
```php
public function show(Task $task)
{
    // Policy で自動認可チェック
    $this->authorize('view', $task);
    
    return view('tasks.show', compact('task'));
}
```

### セキュリティテスト

#### 認証テスト
```php
public function test_user_can_login_with_valid_credentials()
{
    $user = User::factory()->create();
    
    $response = $this->post('/login', [
        'email' => $user->email,
        'password' => 'password',
    ]);
    
    $this->assertAuthenticated();
    $response->assertRedirect('/dashboard');
}
```

#### 認可テスト
```php
public function test_user_cannot_view_other_users_tasks()
{
    $user = User::factory()->create();
    $otherUser = User::factory()->create();
    $task = Task::factory()->for($otherUser)->create();
    
    $response = $this->actingAs($user)->get("/tasks/{$task->id}");
    
    $response->assertStatus(403); // Forbidden
}
```

#### セキュリティテスト
```php
public function test_csrf_protection_is_active()
{
    $response = $this->post('/tasks', [
        'title' => 'Test Task',
        'description' => 'Test Description',
    ]);
    
    $response->assertStatus(419); // CSRF Token Mismatch
}
```

## パフォーマンス最適化

### セッション最適化
```php
// データベースセッション使用（本番環境推奨）
'driver' => 'database',

// セッション有効期限設定
'lifetime' => 120, // 2時間

// 不要セッションの自動削除
php artisan schedule:run
```

### セキュリティログ
```php
// ユーザー活動の記録
UserActivity::create([
    'user_id' => Auth::id(),
    'action' => 'task.created',
    'ip_address' => request()->ip(),
    'user_agent' => request()->userAgent(),
]);
```

## トラブルシューティング

### よくあるエラー

#### 1. CSRF Token Mismatch
```bash
# 解決方法1: キャッシュクリア
php artisan config:clear
php artisan cache:clear

# 解決方法2: CSRFトークンの正しい実装確認
@csrf
```

#### 2. 403 Forbidden (認可エラー)
```bash
# Policy の登録確認
# app/Providers/AuthServiceProvider.php の $policies 配列を確認

# 認可メソッドの実装確認
$this->authorize('view', $task);
```

#### 3. セッションが保持されない
```bash
# セッション設定確認
# config/session.php の設定を確認
# 特に 'domain', 'secure', 'same_site' の設定
```

### デバッグ用コマンド

```bash
# ルート確認（認証ルート含む）
php artisan route:list --name=auth

# ポリシー確認
php artisan tinker
>>> $user = User::find(1);
>>> $task = Task::find(1);
>>> $user->can('view', $task);

# セッション情報確認
php artisan tinker
>>> session()->all();
```

## セキュリティチェックリスト

### ✅ 必須セキュリティ項目

#### 認証・認可
- [ ] Laravel Breeze のセキュアな実装
- [ ] 強力なパスワードポリシーの設定
- [ ] レート制限の実装（ログイン・API）
- [ ] セッション設定の最適化
- [ ] Policy による認可制御

#### 入力値検証・出力
- [ ] Form Request による厳密なバリデーション
- [ ] Blade テンプレートの自動エスケープ活用
- [ ] JSON出力時のXSS対策
- [ ] ファイルアップロード時の検証

#### データベース
- [ ] Eloquent ORM によるSQLインジェクション対策
- [ ] 機密データの暗号化
- [ ] データベース接続の暗号化
- [ ] バックアップデータの暗号化

#### ネットワーク・インフラ
- [ ] HTTPS 強制実装
- [ ] セキュリティヘッダーの実装
- [ ] Content Security Policy (CSP) の設定
- [ ] セキュリティログの記録

## 次のステップ

第6章の内容を理解したら、以下のステップに進みましょう：

1. **第7章: テスト駆動開発入門**
   - 認証・認可機能のテスト実装
   - セキュリティテストの作成
   - Feature Test による包括的テスト

2. **セキュリティ強化の実践**
   - 二要素認証の実装
   - OAuth2 連携の実装
   - 監査ログの充実

3. **パフォーマンス最適化**
   - セッション最適化
   - 認証キャッシュの実装
   - セキュリティヘッダーの最適化

## 参考情報

### 公式ドキュメント
- [Laravel Authentication](https://laravel.com/docs/authentication)
- [Laravel Authorization](https://laravel.com/docs/authorization)
- [Laravel Security](https://laravel.com/docs/security)
- [Laravel Breeze](https://laravel.com/docs/starter-kits#laravel-breeze)

### セキュリティ関連リソース
- [OWASP Top 10](https://owasp.org/www-project-top-ten/)
- [Laravel Security Best Practices](https://laravel-news.com/laravel-security-best-practices)
- [PHP Security Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/PHP_Configuration_Cheat_Sheet.html)

---

**このサンプルコードを通じて、Laravelの強力な認証・セキュリティ機能を実践し、エンタープライズレベルの安全なWebアプリケーション開発スキルを身につけることができます。**