<?php
/**
 * 第6章 サンプルミドルウェア - SecurityHeaders
 * 
 * セキュリティヘッダーを自動追加するミドルウェア
 * - XSS保護ヘッダー
 * - Content Security Policy (CSP)
 * - HTTPS強制ヘッダー
 * - フレーム埋め込み防止
 */

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;

class SecurityHeaders
{
    /**
     * リクエスト処理
     * 
     * @param \Illuminate\Http\Request $request
     * @param \Closure $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);

        // レスポンスオブジェクトでない場合はそのまま返す
        if (!$response instanceof \Symfony\Component\HttpFoundation\Response) {
            return $response;
        }

        // セキュリティヘッダーの追加
        $this->addSecurityHeaders($response, $request);

        return $response;
    }

    /**
     * セキュリティヘッダーの追加
     * 
     * @param \Symfony\Component\HttpFoundation\Response $response
     * @param \Illuminate\Http\Request $request
     */
    private function addSecurityHeaders($response, Request $request): void
    {
        // X-Content-Type-Options: MIME type sniffing 攻撃対策
        $response->headers->set('X-Content-Type-Options', 'nosniff');

        // X-Frame-Options: クリックジャッキング攻撃対策
        $response->headers->set('X-Frame-Options', 'DENY');

        // X-XSS-Protection: XSS攻撃対策（レガシーブラウザ用）
        $response->headers->set('X-XSS-Protection', '1; mode=block');

        // Referrer Policy: リファラー情報の制御
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');

        // Permissions Policy: ブラウザ機能の制御
        $response->headers->set('Permissions-Policy', $this->getPermissionsPolicy());

        // Content Security Policy: XSS攻撃とデータインジェクション攻撃対策
        $csp = $this->buildContentSecurityPolicy($request);
        $response->headers->set('Content-Security-Policy', $csp);

        // HTTPS強制ヘッダー（本番環境のみ）
        if ($this->shouldEnforceHTTPS()) {
            $response->headers->set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');
        }

        // セキュリティヘッダー情報の隠蔽
        $this->removeServerInfo($response);

        // セキュリティログ記録（デバッグ時のみ）
        if (config('app.debug') && config('security.log_headers', false)) {
            $this->logSecurityHeaders($request, $response);
        }
    }

    /**
     * Content Security Policy の構築
     * 
     * @param \Illuminate\Http\Request $request
     * @return string
     */
    private function buildContentSecurityPolicy(Request $request): string
    {
        $policies = [
            // デフォルト: 同一オリジンのみ
            "default-src 'self'",

            // スクリプト: 同一オリジン + 必要なCDN
            "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.tailwindcss.com https://unpkg.com https://cdn.jsdelivr.net",

            // スタイル: 同一オリジン + Googleフォント + インラインスタイル
            "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com https://cdn.tailwindcss.com",

            // フォント: 同一オリジン + Googleフォント
            "font-src 'self' https://fonts.gstatic.com",

            // 画像: 同一オリジン + data URI + HTTPSサイト
            "img-src 'self' data: https: blob:",

            // メディア: 同一オリジンのみ
            "media-src 'self'",

            // オブジェクト: 禁止
            "object-src 'none'",

            // フレーム: 禁止
            "frame-src 'none'",

            // 接続: 同一オリジン（Ajax等）
            "connect-src 'self'",

            // フォーム送信先: 同一オリジンのみ
            "form-action 'self'",

            // ベースURI: 同一オリジンのみ  
            "base-uri 'self'",

            // アップグレード: HTTP を HTTPS にアップグレード
            "upgrade-insecure-requests",
        ];

        // 開発環境での追加設定
        if (app()->environment('local')) {
            // Vite開発サーバー用の設定
            $policies[1] = "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.tailwindcss.com https://unpkg.com https://cdn.jsdelivr.net http://localhost:5173";
            $policies[8] = "connect-src 'self' ws://localhost:5173 http://localhost:5173";
        }

        // 管理者ページでの追加設定
        if ($request->is('admin/*')) {
            // 管理者ページでは一部制限を緩和
            $policies[] = "frame-ancestors 'none'";
        }

        return implode('; ', $policies);
    }

    /**
     * Permissions Policy の取得
     * 
     * @return string
     */
    private function getPermissionsPolicy(): string
    {
        $policies = [
            'accelerometer=()',
            'ambient-light-sensor=()',
            'autoplay=()',
            'battery=()',
            'camera=()',
            'display-capture=()',
            'document-domain=()',
            'encrypted-media=()',
            'fullscreen=(self)',
            'geolocation=()',
            'gyroscope=()',
            'magnetometer=()',
            'microphone=()',
            'midi=()',
            'payment=()',
            'picture-in-picture=()',
            'publickey-credentials-get=()',
            'screen-wake-lock=()',
            'sync-xhr=(self)',
            'usb=()',
            'web-share=()',
            'xr-spatial-tracking=()',
        ];

        return implode(', ', $policies);
    }

    /**
     * HTTPS強制の判定
     * 
     * @return bool
     */
    private function shouldEnforceHTTPS(): bool
    {
        // 本番環境またはHTTPS必須設定がある場合
        return app()->environment('production') || config('security.force_https', false);
    }

    /**
     * サーバー情報の隠蔽
     * 
     * @param \Symfony\Component\HttpFoundation\Response $response
     */
    private function removeServerInfo($response): void
    {
        // サーバー情報を削除（セキュリティ向上）
        $response->headers->remove('Server');
        $response->headers->remove('X-Powered-By');
        
        // Laravel固有のヘッダーも削除
        $response->headers->remove('X-Laravel-Version');
    }

    /**
     * セキュリティヘッダーのログ記録
     * 
     * @param \Illuminate\Http\Request $request
     * @param \Symfony\Component\HttpFoundation\Response $response
     */
    private function logSecurityHeaders(Request $request, $response): void
    {
        $securityHeaders = [
            'Content-Security-Policy',
            'X-Content-Type-Options',
            'X-Frame-Options',
            'X-XSS-Protection',
            'Strict-Transport-Security',
            'Referrer-Policy',
            'Permissions-Policy',
        ];

        $appliedHeaders = [];
        foreach ($securityHeaders as $header) {
            if ($response->headers->has($header)) {
                $appliedHeaders[$header] = $response->headers->get($header);
            }
        }

        Log::debug('セキュリティヘッダー適用', [
            'url' => $request->fullUrl(),
            'method' => $request->method(),
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'headers' => $appliedHeaders,
            'timestamp' => now()->toISOString(),
        ]);
    }

    /**
     * 特定のルートでヘッダーを調整
     * 
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    private function getRouteSpecificHeaders(Request $request): array
    {
        $headers = [];

        // APIエンドポイントの場合
        if ($request->is('api/*')) {
            $headers['Access-Control-Allow-Origin'] = config('cors.allowed_origins', '*');
            $headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, DELETE, OPTIONS';
            $headers['Access-Control-Allow-Headers'] = 'Content-Type, Authorization, X-Requested-With';
        }

        // ファイルダウンロードの場合
        if ($request->is('*/download/*')) {
            $headers['X-Download-Options'] = 'noopen';
            $headers['X-Content-Type-Options'] = 'nosniff';
        }

        return $headers;
    }

    /**
     * レスポンス後の処理
     * 
     * @param \Illuminate\Http\Request $request
     * @param \Symfony\Component\HttpFoundation\Response $response
     */
    public function terminate(Request $request, $response): void
    {
        // セキュリティ違反の検知
        $this->detectSecurityViolations($request, $response);
    }

    /**
     * セキュリティ違反の検知
     * 
     * @param \Illuminate\Http\Request $request
     * @param \Symfony\Component\HttpFoundation\Response $response
     */
    private function detectSecurityViolations(Request $request, $response): void
    {
        // 疑わしいリクエストパターンの検知
        $suspiciousPatterns = [
            '/\<script\>/i',
            '/javascript:/i',
            '/vbscript:/i',
            '/onload=/i',
            '/onerror=/i',
            '/expression\(/i',
            '/document\.cookie/i',
        ];

        $requestContent = $request->getContent();
        $queryString = $request->getQueryString();

        foreach ($suspiciousPatterns as $pattern) {
            if (preg_match($pattern, $requestContent) || preg_match($pattern, $queryString)) {
                Log::warning('疑わしいリクエストを検知', [
                    'pattern' => $pattern,
                    'url' => $request->fullUrl(),
                    'method' => $request->method(),
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'user_id' => auth()->id(),
                ]);
                break;
            }
        }

        // レスポンス時間が異常に長い場合（DoS攻撃の可能性）
        if (defined('LARAVEL_START')) {
            $responseTime = microtime(true) - LARAVEL_START;
            if ($responseTime > 30) { // 30秒以上
                Log::warning('異常に長いレスポンス時間', [
                    'response_time' => $responseTime,
                    'url' => $request->fullUrl(),
                    'method' => $request->method(),
                    'ip' => $request->ip(),
                ]);
            }
        }
    }
}

/*
使用方法：

1. ミドルウェアの登録（app/Http/Kernel.php）:
   
   protected $middleware = [
       // ... 他のミドルウェア
       \App\Http\Middleware\SecurityHeaders::class,
   ];

2. 設定ファイル（config/security.php）の作成:
   
   return [
       'force_https' => env('FORCE_HTTPS', false),
       'log_headers' => env('LOG_SECURITY_HEADERS', false),
   ];

3. 環境変数（.env）の設定:
   
   FORCE_HTTPS=true
   LOG_SECURITY_HEADERS=false

4. 特定のルートグループでの使用:
   
   Route::middleware(['security.headers'])->group(function () {
       // セキュリティヘッダーが適用されるルート
   });

セキュリティ効果：
- XSS攻撃の防御
- クリックジャッキング攻撃の防御  
- MIME type sniffing 攻撃の防御
- リファラー情報の漏洩防止
- HTTPS通信の強制
- 不要なブラウザ機能の無効化
*/