<?php
/**
 * 第6章 サンプルポリシー - TaskPolicy
 * 
 * タスクに対するアクセス制御ポリシー
 * - ユーザー所有権ベースの認可制御
 * - 共有機能の実装
 * - 管理者権限の考慮
 */

namespace App\Policies;

use App\Models\Task;
use App\Models\User;
use Illuminate\Auth\Access\Response;
use Illuminate\Support\Facades\Log;

class TaskPolicy
{
    /**
     * タスク一覧表示権限
     * 
     * ✅ セキュリティポイント：
     * - 認証済みユーザーは自分のタスク一覧を表示可能
     * - ゲストユーザーはアクセス不可
     */
    public function viewAny(User $user): bool
    {
        // 認証済みユーザーは自分のタスク一覧を表示可能
        return true;
    }

    /**
     * 個別タスク表示権限
     * 
     * ✅ セキュリティポイント：
     * - タスクの所有者のみ表示可能
     * - 共有設定されたタスクの場合は共有相手も表示可能
     * - 管理者は全てのタスクを表示可能
     */
    public function view(User $user, Task $task): Response
    {
        // 管理者は全てのタスクを閲覧可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.view', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限によりアクセスが許可されました。');
        }

        // 自分のタスクの場合
        if ($task->user_id === $user->id) {
            $this->logPolicyCheck('task.view', $user, $task, 'allowed_owner');
            return Response::allow('タスクの所有者としてアクセスが許可されました。');
        }

        // 共有されたタスクの場合
        if (!$task->is_private && $task->shared_with) {
            $sharedUsers = json_decode($task->shared_with, true) ?? [];
            if (in_array($user->id, $sharedUsers)) {
                $this->logPolicyCheck('task.view', $user, $task, 'allowed_shared');
                return Response::allow('共有タスクとしてアクセスが許可されました。');
            }
        }

        // パブリックタスクの場合（将来の機能拡張用）
        if ($task->is_public ?? false) {
            $this->logPolicyCheck('task.view', $user, $task, 'allowed_public');
            return Response::allow('パブリックタスクとしてアクセスが許可されました。');
        }

        // アクセス拒否
        $this->logPolicyCheck('task.view', $user, $task, 'denied');
        return Response::deny('このタスクにアクセスする権限がありません。');
    }

    /**
     * タスク作成権限
     * 
     * ✅ セキュリティポイント：
     * - 認証済みユーザーは作成可能
     * - アカウント制限の考慮
     */
    public function create(User $user): Response
    {
        // アカウントがアクティブかチェック
        if (!$user->isActive()) {
            return Response::deny('アカウントが無効化されているため、タスクを作成できません。');
        }

        // タスク数制限のチェック（有料プランなどの実装を想定）
        if ($this->hasReachedTaskLimit($user)) {
            return Response::deny('タスク数の上限に達しています。プランをアップグレードしてください。');
        }

        $this->logPolicyCheck('task.create', $user, null, 'allowed');
        return Response::allow('タスク作成が許可されました。');
    }

    /**
     * タスク更新権限
     * 
     * ✅ セキュリティポイント：
     * - 所有者のみ更新可能
     * - 管理者は全てのタスクを更新可能
     * - 共有タスクの編集権限チェック
     */
    public function update(User $user, Task $task): Response
    {
        // 管理者は全てのタスクを更新可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.update', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限により更新が許可されました。');
        }

        // タスクの所有者のみ更新可能
        if ($task->user_id === $user->id) {
            $this->logPolicyCheck('task.update', $user, $task, 'allowed_owner');
            return Response::allow('所有者として更新が許可されました。');
        }

        // 共有タスクで編集権限がある場合
        if ($this->hasSharedEditPermission($user, $task)) {
            $this->logPolicyCheck('task.update', $user, $task, 'allowed_shared_editor');
            return Response::allow('共有編集者として更新が許可されました。');
        }

        $this->logPolicyCheck('task.update', $user, $task, 'denied');
        return Response::deny('このタスクを更新する権限がありません。');
    }

    /**
     * タスク削除権限
     * 
     * ✅ セキュリティポイント：
     * - 所有者のみ削除可能
     * - 管理者は全てのタスクを削除可能
     * - 共有タスクは削除不可
     */
    public function delete(User $user, Task $task): Response
    {
        // 管理者は全てのタスクを削除可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.delete', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限により削除が許可されました。');
        }

        // タスクの所有者のみ削除可能
        if ($task->user_id === $user->id) {
            $this->logPolicyCheck('task.delete', $user, $task, 'allowed_owner');
            return Response::allow('所有者として削除が許可されました。');
        }

        $this->logPolicyCheck('task.delete', $user, $task, 'denied');
        return Response::deny('このタスクを削除する権限がありません。');
    }

    /**
     * タスク復元権限
     * 
     * ✅ セキュリティポイント：
     * - 削除されたタスクの復元権限
     * - 所有者と管理者のみ復元可能
     */
    public function restore(User $user, Task $task): Response
    {
        // 管理者は全てのタスクを復元可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.restore', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限により復元が許可されました。');
        }

        // タスクの所有者のみ復元可能
        if ($task->user_id === $user->id) {
            $this->logPolicyCheck('task.restore', $user, $task, 'allowed_owner');
            return Response::allow('所有者として復元が許可されました。');
        }

        $this->logPolicyCheck('task.restore', $user, $task, 'denied');
        return Response::deny('このタスクを復元する権限がありません。');
    }

    /**
     * タスク完全削除権限
     * 
     * ✅ セキュリティポイント：
     * - 物理削除は非常に危険な操作
     * - 管理者のみ実行可能
     */
    public function forceDelete(User $user, Task $task): Response
    {
        // 管理者のみ完全削除可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.force_delete', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限により完全削除が許可されました。');
        }

        $this->logPolicyCheck('task.force_delete', $user, $task, 'denied');
        return Response::deny('タスクを完全削除する権限がありません。');
    }

    /**
     * タスク共有権限
     * 
     * ✅ セキュリティポイント：
     * - 所有者のみ共有設定可能
     * - プライベートタスクの共有制御
     */
    public function share(User $user, Task $task): Response
    {
        // 管理者は全てのタスクを共有設定可能
        if ($user->isAdmin()) {
            $this->logPolicyCheck('task.share', $user, $task, 'allowed_admin');
            return Response::allow('管理者権限により共有設定が許可されました。');
        }

        // タスクの所有者のみ共有設定可能
        if ($task->user_id === $user->id) {
            $this->logPolicyCheck('task.share', $user, $task, 'allowed_owner');
            return Response::allow('所有者として共有設定が許可されました。');
        }

        $this->logPolicyCheck('task.share', $user, $task, 'denied');
        return Response::deny('このタスクの共有設定を変更する権限がありません。');
    }

    /**
     * 一括操作権限
     * 
     * ✅ セキュリティポイント：
     * - 一括操作の危険性を考慮
     * - 所有者のタスクのみ一括操作可能
     */
    public function bulkAction(User $user): Response
    {
        // アクティブなユーザーのみ一括操作可能
        if (!$user->isActive()) {
            return Response::deny('アカウントが無効化されているため、一括操作できません。');
        }

        // 一括操作の実行制限（レート制限など）
        if ($this->hasExceededBulkActionLimit($user)) {
            return Response::deny('一括操作の制限に達しています。しばらく時間を置いてから再実行してください。');
        }

        $this->logPolicyCheck('task.bulk_action', $user, null, 'allowed');
        return Response::allow('一括操作が許可されました。');
    }

    /**
     * タスク数制限チェック
     * 
     * @param User $user
     * @return bool
     */
    private function hasReachedTaskLimit(User $user): bool
    {
        $currentTaskCount = $user->tasks()->count();
        $taskLimit = $user->getTaskLimit(); // ユーザーのプランに応じた上限

        return $currentTaskCount >= $taskLimit;
    }

    /**
     * 共有編集権限チェック
     * 
     * @param User $user
     * @param Task $task
     * @return bool
     */
    private function hasSharedEditPermission(User $user, Task $task): bool
    {
        if ($task->is_private || !$task->shared_with) {
            return false;
        }

        $sharedData = json_decode($task->shared_with, true) ?? [];
        
        // 共有データの形式: ['user_id' => ['view', 'edit']] または単純な user_id 配列
        if (isset($sharedData[$user->id])) {
            $permissions = $sharedData[$user->id];
            return is_array($permissions) ? in_array('edit', $permissions) : false;
        }

        return false;
    }

    /**
     * 一括操作制限チェック
     * 
     * @param User $user
     * @return bool
     */
    private function hasExceededBulkActionLimit(User $user): bool
    {
        // 過去1時間の一括操作回数をチェック
        $recentBulkActions = \DB::table('user_activities')
            ->where('user_id', $user->id)
            ->where('action', 'task.bulk_action')
            ->where('created_at', '>=', now()->subHour())
            ->count();

        return $recentBulkActions >= 10; // 1時間に10回まで
    }

    /**
     * ポリシーチェックのログ記録
     * 
     * @param string $action
     * @param User $user
     * @param Task|null $task
     * @param string $result
     */
    private function logPolicyCheck(string $action, User $user, ?Task $task, string $result): void
    {
        Log::info('タスクポリシーチェック実行', [
            'action' => $action,
            'user_id' => $user->id,
            'task_id' => $task?->id,
            'task_user_id' => $task?->user_id,
            'result' => $result,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'timestamp' => now()->toISOString(),
        ]);

        // セキュリティ警告が必要な場合
        if ($result === 'denied' && $task) {
            Log::warning('タスクへの不正アクセス試行', [
                'action' => $action,
                'user_id' => $user->id,
                'task_id' => $task->id,
                'task_owner_id' => $task->user_id,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);
        }
    }

    /**
     * エラーハンドリング用のフォールバックメソッド
     * 
     * 定義されていないメソッドへのアクセスを拒否
     */
    public function __call($name, $arguments): Response
    {
        Log::warning('未定義のポリシーメソッドへのアクセス', [
            'method' => $name,
            'user_id' => $arguments[0]?->id ?? null,
            'arguments' => count($arguments),
        ]);

        return Response::deny('この操作は許可されていません。');
    }
}