# 第7章 サンプルコード - テスト駆動開発入門

## 概要

第7章「テスト駆動開発入門」で使用するサンプルコード集です。
Laravel テストフレームワークを使用した包括的なテスト実装から、Feature Test による機能テスト、API テストの実装まで、品質の高い Web アプリケーション開発に不可欠なテスト技術を学習します。

## ファイル構成

```
chapter07/
├── tests/                               # テストファイル
│   ├── Unit/                           # 単体テスト
│   │   ├── TaskTest.php               # タスクモデルの単体テスト
│   │   ├── UserTest.php               # ユーザーモデルの単体テスト
│   │   └── CategoryTest.php           # カテゴリモデルの単体テスト
│   ├── Feature/                        # 機能テスト
│   │   ├── AuthenticationTest.php     # 認証機能テスト
│   │   ├── TaskManagementTest.php     # タスク管理機能テスト
│   │   ├── SecurityTest.php           # セキュリティ機能テスト
│   │   └── TaskApiTest.php            # タスクAPI機能テスト
│   └── TestCase.php                   # テスト基底クラス
├── config/                             # テスト設定ファイル
│   ├── database-testing.php          # テスト用データベース設定
│   └── phpunit-example.xml           # PHPUnit設定例
├── phpunit/                           # PHPUnit関連ファイル
│   ├── bootstrap.php                  # テストブートストラップ
│   └── coverage-report.html          # カバレッジレポート例
├── factories/                         # テスト用ファクトリー
│   ├── TaskFactoryExtended.php       # 拡張タスクファクトリー
│   └── UserFactoryExtended.php       # 拡張ユーザーファクトリー
├── raw-php-examples/                  # 生PHP比較用テスト例
│   ├── basic-php-test.php            # 基本的な生PHPテスト例
│   └── database-test-raw.php         # データベーステスト生PHP例
├── github-actions/                    # CI/CD設定
│   └── tests.yml                     # GitHub Actions テスト設定
└── README.md                         # このファイル
```

## セットアップ手順

### 1. テスト環境の準備

```bash
# テスト用データベース設定（SQLite in-memory）
# .env.testing ファイルを作成
cp .env .env.testing

# .env.testing の内容を編集
DB_CONNECTION=testing
DB_DATABASE=:memory:
CACHE_DRIVER=array
QUEUE_CONNECTION=sync
SESSION_DRIVER=array
MAIL_MAILER=array
```

### 2. テストファイルの配置

#### 単体テストファイルの配置
```bash
# Unit テストをコピー
cp tests/Unit/*.php tests/Unit/
```

#### 機能テストファイルの配置
```bash
# Feature テストをコピー
cp tests/Feature/*.php tests/Feature/
```

#### テスト基底クラスの配置
```bash
# TestCase をコピー
cp tests/TestCase.php tests/TestCase.php
```

### 3. PHPUnit設定

```bash
# phpunit.xml を適切に配置
cp config/phpunit-example.xml phpunit.xml

# テスト実行権限の設定
chmod +x vendor/bin/phpunit
```

### 4. 基本的なテスト実行

```bash
# 全テストの実行
php artisan test

# または直接PHPUnitを使用
./vendor/bin/phpunit

# 詳細出力でテスト実行
php artisan test --verbose

# カバレッジ付きでテスト実行
php artisan test --coverage
```

## 実装内容詳細

### Laravel テストフレームワーク

#### テストの種類と特徴

1. **Unit Test（単体テスト）**
   - モデル・メソッドの個別機能をテスト
   - データベースアクセスを含まない軽量テスト
   - 高速実行・独立性を重視

2. **Feature Test（機能テスト）**
   - HTTPリクエスト・レスポンスをテスト
   - 実際のユーザー操作を再現
   - データベース・セッション・認証を含む統合テスト

#### テスト環境の自動化
```php
// 自動データベースリセット
use RefreshDatabase;

// 自動ファクトリー使用
$user = User::factory()->create();

// 自動認証ユーザー設定
$this->actingAs($user);
```

### 包括的なテスト実装

#### 認証機能のテスト
```php
// ログイン成功テスト
public function test_users_can_authenticate(): void
{
    $user = User::factory()->create([
        'email' => 'test@example.com',
        'password' => Hash::make('password'),
    ]);

    $response = $this->post('/login', [
        'email' => 'test@example.com',
        'password' => 'password',
    ]);

    $this->assertAuthenticated();
    $response->assertRedirect('/dashboard');
}
```

#### セキュリティ機能のテスト
```php
// CSRF保護テスト
public function test_csrf_protection_is_active(): void
{
    $response = $this->post('/tasks', ['title' => 'Test']);
    $response->assertStatus(419); // CSRF Token Mismatch
}

// XSS保護テスト
public function test_xss_protection(): void
{
    $malicious = '<script>alert("XSS")</script>';
    // エスケープされることを確認
    $response->assertSee('&lt;script&gt;', false);
}
```

#### API テスト
```php
// API認証テスト
Sanctum::actingAs($user);
$response = $this->getJson('/api/tasks');
$response->assertStatus(200)
         ->assertJsonStructure(['data', 'links', 'meta']);
```

### テスト効率化機能

#### テストファクトリーの活用
```php
// 複雑なテストデータの生成
$tasks = Task::factory()
    ->count(10)
    ->for($user)
    ->highPriority()
    ->overdue()
    ->create();
```

#### カスタムアサーション
```php
// 独自アサーションメソッド
protected function assertDatabaseHasAll($table, $data)
{
    foreach ($data as $attributes) {
        $this->assertDatabaseHas($table, $attributes);
    }
}
```

#### テスト並列実行
```bash
# 複数プロセスでテスト高速化
php artisan test --parallel --processes=4
```

### 継続的インテグレーション

#### GitHub Actions 設定
```yaml
# 自動テスト実行
- name: Execute tests
  run: php artisan test --coverage --min=80
```

#### テストカバレッジ
```bash
# カバレッジレポート生成
php artisan test --coverage-html coverage-report

# 最小カバレッジ率の指定
php artisan test --coverage --min=80
```

## 使用例とパターン

### 基本的なテストパターン

#### 1. データベーステスト
```php
public function test_user_can_create_task(): void
{
    $user = User::factory()->create();
    $category = Category::factory()->for($user)->create();
    
    $response = $this->actingAs($user)->post('/tasks', [
        'title' => 'テストタスク',
        'category_id' => $category->id,
        'priority' => 'high',
    ]);
    
    $response->assertRedirect();
    $this->assertDatabaseHas('tasks', [
        'title' => 'テストタスク',
        'user_id' => $user->id,
    ]);
}
```

#### 2. バリデーションテスト
```php
public function test_task_creation_validates_required_fields(): void
{
    $user = User::factory()->create();
    
    $response = $this->actingAs($user)->post('/tasks', [
        'title' => '', // 必須項目が空
        'priority' => 'invalid', // 不正な値
    ]);
    
    $response->assertSessionHasErrors(['title', 'priority']);
}
```

#### 3. 認可テスト
```php
public function test_user_cannot_access_other_users_tasks(): void
{
    $user1 = User::factory()->create();
    $user2 = User::factory()->create();
    $task = Task::factory()->for($user2)->create();
    
    $response = $this->actingAs($user1)->get("/tasks/{$task->id}");
    
    $response->assertStatus(403); // Forbidden
}
```

### 高度なテストパターン

#### 1. モックとスタブ
```php
public function test_email_notification_is_sent(): void
{
    Mail::fake();
    
    $user = User::factory()->create();
    $task = Task::factory()->for($user)->create(['due_date' => now()->addDay()]);
    
    // 期限通知処理の実行
    $this->artisan('tasks:send-due-notifications');
    
    Mail::assertSent(TaskDueNotification::class);
}
```

#### 2. イベントテスト
```php
public function test_task_creation_fires_event(): void
{
    Event::fake();
    
    $user = User::factory()->create();
    
    $this->actingAs($user)->post('/tasks', [
        'title' => 'テストタスク',
        // ... その他のデータ
    ]);
    
    Event::assertDispatched(TaskCreated::class);
}
```

#### 3. ジョブテスト
```php
public function test_task_deletion_queues_cleanup_job(): void
{
    Queue::fake();
    
    $task = Task::factory()->create();
    
    $this->actingAs($task->user)->delete("/tasks/{$task->id}");
    
    Queue::assertPushed(CleanupTaskData::class);
}
```

## パフォーマンス最適化

### テスト実行速度の向上

#### 1. データベース最適化
```php
// SQLite in-memory の使用
'testing' => [
    'driver' => 'sqlite',
    'database' => ':memory:',
],
```

#### 2. 並列テスト実行
```bash
# 4プロセスで並列実行
php artisan test --parallel --processes=4
```

#### 3. テストスイート分割
```bash
# Unit テストのみ実行（高速）
php artisan test --testsuite=Unit

# Feature テストのみ実行
php artisan test --testsuite=Feature
```

### メモリ使用量の最適化

#### トレイトの適切な使用
```php
class TaskTest extends TestCase
{
    use RefreshDatabase;    // テスト毎にDB初期化
    use WithFaker;         // フェイクデータ生成
    
    // テストメソッド...
}
```

## トラブルシューティング

### よくあるエラー

#### 1. テスト環境でのデータベース接続エラー
```bash
# 解決方法：.env.testing の確認
DB_CONNECTION=testing
DB_DATABASE=:memory:

# または
php artisan config:clear --env=testing
```

#### 2. ファクトリーエラー
```bash
# 解決方法：ファクトリーの登録確認
php artisan tinker
>>> User::factory()->create()
```

#### 3. 認証テストエラー
```bash
# 解決方法：セッション設定確認
SESSION_DRIVER=array
```

### デバッグ用コマンド

```bash
# 詳細出力でテスト実行
php artisan test --verbose

# 特定のテストのみ実行
php artisan test --filter=test_user_can_login

# デバッグ情報付きで実行
php artisan test --debug

# 失敗時に停止
php artisan test --stop-on-failure
```

## テスト品質の評価

### コードカバレッジの測定

```bash
# HTMLレポート生成
php artisan test --coverage-html coverage-report

# コンソールでカバレッジ表示
php artisan test --coverage

# 最小カバレッジ率指定
php artisan test --coverage --min=80
```

### テスト品質メトリクス

1. **カバレッジ率**: 80%以上を目標
2. **テスト実行時間**: 全体で5分以内
3. **テスト独立性**: 各テストが他に依存しない
4. **テスト可読性**: 目的と期待結果が明確

## 生PHP vs Laravel テスト比較

### 実装工数比較

| テスト機能 | 生PHP | Laravel | 削減率 |
|------------|-------|---------|--------|
| **環境構築** | 6-10時間 | 30分 | 95% |
| **Unit Test** | 8-12時間 | 1-2時間 | 85% |
| **HTTP Test** | 15-25時間 | 3-5時間 | 80% |
| **DB Test** | 12-18時間 | 2-3時間 | 80% |
| **認証Test** | 10-15時間 | 2-4時間 | 75% |
| **API Test** | 20-30時間 | 4-6時間 | 80% |
| **CI/CD** | 8-15時間 | 1-2時間 | 90% |

**総工数：生PHP 79-125時間 → Laravel 13.5-22.5時間（削減率：82-83%）**

### 機能品質比較

| 項目 | 生PHP | Laravel | 改善度 |
|------|-------|---------|--------|
| **テスト環境分離** | 困難 | 自動 | ◎ |
| **データベーステスト** | 複雑 | 簡単 | ◎ |
| **HTTP テスト** | 困難 | 簡単 | ◎ |
| **モック・スタブ** | 手動実装 | 自動 | ◎ |
| **継続的インテグレーション** | 複雑設定 | 簡単設定 | ◎ |
| **カバレッジ測定** | 困難 | ワンコマンド | ◎ |

## 次のステップ

第7章の内容を理解したら、以下のステップに進みましょう：

1. **第8章: 本番環境へのデプロイ**
   - テストパイプラインと連携したデプロイ
   - 本番環境でのテスト戦略
   - モニタリングと品質管理

2. **高度なテスト技術**
   - End-to-End テスト（Laravel Dusk）
   - パフォーマンステスト
   - セキュリティテスト

3. **テスト自動化の拡張**
   - 複数ブラウザでのテスト
   - モバイル対応テスト
   - 負荷テスト

## 参考情報

### 公式ドキュメント
- [Laravel Testing](https://laravel.com/docs/testing)
- [Laravel HTTP Tests](https://laravel.com/docs/http-tests)
- [Laravel Database Testing](https://laravel.com/docs/database-testing)
- [PHPUnit Documentation](https://phpunit.de/documentation.html)

### 追加学習リソース
- [Test-Driven Development in Laravel](https://laracasts.com/series/build-a-laravel-app-with-tdd)
- [Laravel Testing Best Practices](https://github.com/alexeymezenin/laravel-best-practices)
- [Laravel Dusk Browser Testing](https://laravel.com/docs/dusk)

---

**このサンプルコードを通じて、Laravel の強力なテストフレームワークを実践し、テスト駆動開発（TDD）による高品質な Web アプリケーション開発スキルを身につけることができます。**