<?php
/**
 * 第7章 サンプルテスト - TaskTest（Unit Test）
 * 
 * タスクモデルの単体テスト
 * - モデルの基本機能テスト
 * - リレーションのテスト
 * - スコープのテスト
 * - カスタムメソッドのテスト
 */

namespace Tests\Unit;

use App\Models\Task;
use App\Models\User;
use App\Models\Category;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Carbon\Carbon;

class TaskTest extends TestCase
{
    use RefreshDatabase;

    /**
     * タスクの基本的な属性が正しく設定されることをテスト
     */
    public function test_task_has_required_attributes(): void
    {
        $user = User::factory()->create();
        $category = Category::factory()->for($user)->create();
        
        $task = Task::factory()->create([
            'user_id' => $user->id,
            'category_id' => $category->id,
            'title' => 'テストタスク',
            'description' => 'テスト用の説明文です。',
            'priority' => 'high',
            'completed' => false,
            'due_date' => '2024-12-31',
            'is_private' => true,
        ]);

        // 基本属性の確認
        $this->assertEquals('テストタスク', $task->title);
        $this->assertEquals('テスト用の説明文です。', $task->description);
        $this->assertEquals('high', $task->priority);
        $this->assertFalse($task->completed);
        $this->assertEquals('2024-12-31', $task->due_date->format('Y-m-d'));
        $this->assertTrue($task->is_private);
        
        // 外部キーの確認
        $this->assertEquals($user->id, $task->user_id);
        $this->assertEquals($category->id, $task->category_id);
    }

    /**
     * タスクとユーザーのリレーションが正しく動作することをテスト
     */
    public function test_task_belongs_to_user(): void
    {
        $user = User::factory()->create(['name' => 'テストユーザー']);
        $task = Task::factory()->for($user)->create();

        // リレーション取得の確認
        $this->assertInstanceOf(User::class, $task->user);
        $this->assertEquals('テストユーザー', $task->user->name);
        $this->assertEquals($user->id, $task->user->id);
    }

    /**
     * タスクとカテゴリのリレーションが正しく動作することをテスト
     */
    public function test_task_belongs_to_category(): void
    {
        $category = Category::factory()->create(['name' => 'テストカテゴリ']);
        $task = Task::factory()->for($category)->create();

        // リレーション取得の確認
        $this->assertInstanceOf(Category::class, $task->category);
        $this->assertEquals('テストカテゴリ', $task->category->name);
        $this->assertEquals($category->id, $task->category->id);
    }

    /**
     * ユーザーからタスクへの逆リレーションが正しく動作することをテスト
     */
    public function test_user_has_many_tasks(): void
    {
        $user = User::factory()->create();
        $tasks = Task::factory(3)->for($user)->create();

        // hasMany リレーションの確認
        $this->assertCount(3, $user->tasks);
        $this->assertInstanceOf(Task::class, $user->tasks->first());
        
        // 各タスクが正しいユーザーに属していることを確認
        $user->tasks->each(function ($task) use ($user) {
            $this->assertEquals($user->id, $task->user_id);
        });
    }

    /**
     * completed スコープが正しく動作することをテスト
     */
    public function test_completed_scope_returns_only_completed_tasks(): void
    {
        // 完了済みと未完了のタスクを作成
        Task::factory(3)->create(['completed' => true]);
        Task::factory(2)->create(['completed' => false]);

        $completedTasks = Task::completed()->get();
        
        // 完了済みタスクのみが返されることを確認
        $this->assertCount(3, $completedTasks);
        $completedTasks->each(function ($task) {
            $this->assertTrue($task->completed);
        });
    }

    /**
     * pending スコープが正しく動作することをテスト
     */
    public function test_pending_scope_returns_only_pending_tasks(): void
    {
        Task::factory(3)->create(['completed' => true]);
        Task::factory(4)->create(['completed' => false]);

        $pendingTasks = Task::pending()->get();
        
        $this->assertCount(4, $pendingTasks);
        $pendingTasks->each(function ($task) {
            $this->assertFalse($task->completed);
        });
    }

    /**
     * highPriority スコープが正しく動作することをテスト
     */
    public function test_high_priority_scope_returns_only_high_priority_tasks(): void
    {
        Task::factory(2)->create(['priority' => 'high']);
        Task::factory(3)->create(['priority' => 'medium']);
        Task::factory(1)->create(['priority' => 'low']);

        $highPriorityTasks = Task::highPriority()->get();
        
        $this->assertCount(2, $highPriorityTasks);
        $highPriorityTasks->each(function ($task) {
            $this->assertEquals('high', $task->priority);
        });
    }

    /**
     * dueToday スコープが正しく動作することをテスト
     */
    public function test_due_today_scope_returns_tasks_due_today(): void
    {
        Task::factory()->create(['due_date' => now()->startOfDay()]);
        Task::factory()->create(['due_date' => now()->addDay()]);
        Task::factory()->create(['due_date' => now()->subDay()]);

        $tasksDueToday = Task::dueToday()->get();
        
        $this->assertCount(1, $tasksDueToday);
        $this->assertTrue($tasksDueToday->first()->due_date->isToday());
    }

    /**
     * overdue スコープが正しく動作することをテスト
     */
    public function test_overdue_scope_returns_overdue_incomplete_tasks(): void
    {
        // 期限切れの未完了タスク
        Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => false,
        ]);
        
        // 期限切れの完了済みタスク（スコープに含まれない）
        Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => true,
        ]);
        
        // 期限未到来の未完了タスク（スコープに含まれない）
        Task::factory()->create([
            'due_date' => now()->addDays(1),
            'completed' => false,
        ]);

        $overdueTasks = Task::overdue()->get();
        
        $this->assertCount(1, $overdueTasks);
        $overdueTasks->each(function ($task) {
            $this->assertTrue($task->due_date->isPast());
            $this->assertFalse($task->completed);
        });
    }

    /**
     * タスクの完了状態を切り替えるメソッドのテスト
     */
    public function test_task_can_be_marked_as_completed(): void
    {
        $task = Task::factory()->create([
            'completed' => false,
            'completed_at' => null,
        ]);
        
        // 完了にマーク
        $task->markAsCompleted();
        
        $this->assertTrue($task->completed);
        $this->assertNotNull($task->completed_at);
        $this->assertInstanceOf(Carbon::class, $task->completed_at);
    }

    /**
     * タスクの未完了状態を切り替えるメソッドのテスト
     */
    public function test_task_can_be_marked_as_pending(): void
    {
        $task = Task::factory()->create([
            'completed' => true,
            'completed_at' => now(),
        ]);
        
        // 未完了にマーク
        $task->markAsPending();
        
        $this->assertFalse($task->completed);
        $this->assertNull($task->completed_at);
    }

    /**
     * タスクが期限切れかどうかを判定するメソッドのテスト
     */
    public function test_task_can_be_identified_as_overdue(): void
    {
        // 期限切れの未完了タスク
        $overdueTask = Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => false,
        ]);

        // 期限未到来のタスク
        $futureTask = Task::factory()->create([
            'due_date' => now()->addDays(1),
            'completed' => false,
        ]);

        // 期限切れだが完了済みのタスク
        $completedTask = Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => true,
        ]);

        // 期限なしのタスク
        $noDateTask = Task::factory()->create([
            'due_date' => null,
            'completed' => false,
        ]);

        $this->assertTrue($overdueTask->isOverdue());
        $this->assertFalse($futureTask->isOverdue());
        $this->assertFalse($completedTask->isOverdue());
        $this->assertFalse($noDateTask->isOverdue());
    }

    /**
     * タスクが今日期限かどうかを判定するメソッドのテスト
     */
    public function test_task_can_be_identified_as_due_today(): void
    {
        $todayTask = Task::factory()->create([
            'due_date' => now()->startOfDay(),
        ]);

        $tomorrowTask = Task::factory()->create([
            'due_date' => now()->addDay(),
        ]);

        $noDateTask = Task::factory()->create([
            'due_date' => null,
        ]);

        $this->assertTrue($todayTask->isDueToday());
        $this->assertFalse($tomorrowTask->isDueToday());
        $this->assertFalse($noDateTask->isDueToday());
    }

    /**
     * タスクの残り日数を取得するメソッドのテスト
     */
    public function test_task_returns_correct_days_until_due(): void
    {
        $taskInThreeDays = Task::factory()->create([
            'due_date' => now()->addDays(3),
        ]);

        $taskYesterday = Task::factory()->create([
            'due_date' => now()->subDays(1),
        ]);

        $taskToday = Task::factory()->create([
            'due_date' => now(),
        ]);

        $noDateTask = Task::factory()->create([
            'due_date' => null,
        ]);

        $this->assertEquals(3, $taskInThreeDays->daysUntilDue());
        $this->assertEquals(-1, $taskYesterday->daysUntilDue());
        $this->assertEquals(0, $taskToday->daysUntilDue());
        $this->assertNull($noDateTask->daysUntilDue());
    }

    /**
     * タスクの優先度を数値で取得するメソッドのテスト
     */
    public function test_task_returns_correct_priority_value(): void
    {
        $highTask = Task::factory()->create(['priority' => 'high']);
        $mediumTask = Task::factory()->create(['priority' => 'medium']);
        $lowTask = Task::factory()->create(['priority' => 'low']);

        $this->assertEquals(3, $highTask->getPriorityValue());
        $this->assertEquals(2, $mediumTask->getPriorityValue());
        $this->assertEquals(1, $lowTask->getPriorityValue());
    }

    /**
     * タスクのタグを配列で取得するメソッドのテスト
     */
    public function test_task_returns_tags_as_array(): void
    {
        $taskWithTags = Task::factory()->create([
            'tags' => json_encode(['laravel', 'php', 'testing']),
        ]);

        $taskWithoutTags = Task::factory()->create([
            'tags' => null,
        ]);

        $this->assertEquals(['laravel', 'php', 'testing'], $taskWithTags->getTagsArray());
        $this->assertEquals([], $taskWithoutTags->getTagsArray());
    }

    /**
     * タスクのタグを文字列で取得するメソッドのテスト
     */
    public function test_task_returns_tags_as_string(): void
    {
        $taskWithTags = Task::factory()->create([
            'tags' => json_encode(['laravel', 'php', 'testing']),
        ]);

        $taskWithoutTags = Task::factory()->create([
            'tags' => null,
        ]);

        $this->assertEquals('laravel, php, testing', $taskWithTags->getTagsString());
        $this->assertEquals('', $taskWithoutTags->getTagsString());
    }

    /**
     * タスクの表示用ステータスを取得するメソッドのテスト
     */
    public function test_task_returns_correct_status_label(): void
    {
        $completedTask = Task::factory()->create(['completed' => true]);
        $pendingTask = Task::factory()->create(['completed' => false]);

        $this->assertEquals('完了', $completedTask->getStatusLabel());
        $this->assertEquals('進行中', $pendingTask->getStatusLabel());
    }

    /**
     * タスクの優先度ラベルを取得するメソッドのテスト
     */
    public function test_task_returns_correct_priority_label(): void
    {
        $highTask = Task::factory()->create(['priority' => 'high']);
        $mediumTask = Task::factory()->create(['priority' => 'medium']);
        $lowTask = Task::factory()->create(['priority' => 'low']);

        $this->assertEquals('高', $highTask->getPriorityLabel());
        $this->assertEquals('中', $mediumTask->getPriorityLabel());
        $this->assertEquals('低', $lowTask->getPriorityLabel());
    }

    /**
     * タスクの期限状況を示すCSSクラスを取得するメソッドのテスト
     */
    public function test_task_returns_correct_due_date_css_class(): void
    {
        $overdueTask = Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => false,
        ]);

        $todayTask = Task::factory()->create([
            'due_date' => now(),
            'completed' => false,
        ]);

        $soonTask = Task::factory()->create([
            'due_date' => now()->addDays(2),
            'completed' => false,
        ]);

        $futureTask = Task::factory()->create([
            'due_date' => now()->addWeeks(2),
            'completed' => false,
        ]);

        $completedTask = Task::factory()->create([
            'due_date' => now()->subDays(1),
            'completed' => true,
        ]);

        $this->assertEquals('text-red-600', $overdueTask->getDueDateCssClass());
        $this->assertEquals('text-orange-600', $todayTask->getDueDateCssClass());
        $this->assertEquals('text-yellow-600', $soonTask->getDueDateCssClass());
        $this->assertEquals('text-green-600', $futureTask->getDueDateCssClass());
        $this->assertEquals('text-gray-600', $completedTask->getDueDateCssClass());
    }

    /**
     * Fillable属性が正しく設定されていることをテスト
     */
    public function test_task_has_correct_fillable_attributes(): void
    {
        $task = new Task();
        $expectedFillable = [
            'title',
            'description',
            'priority',
            'due_date',
            'completed',
            'completed_at',
            'is_private',
            'tags',
            'user_id',
            'category_id',
        ];

        $this->assertEquals($expectedFillable, $task->getFillable());
    }

    /**
     * Castが正しく設定されていることをテスト
     */
    public function test_task_has_correct_casts(): void
    {
        $task = new Task();
        $expectedCasts = [
            'id' => 'int',
            'due_date' => 'date',
            'completed' => 'boolean',
            'completed_at' => 'datetime',
            'is_private' => 'boolean',
        ];

        $this->assertEquals($expectedCasts, $task->getCasts());
    }

    /**
     * ソフトデリートが正しく動作することをテスト
     */
    public function test_task_uses_soft_deletes(): void
    {
        $task = Task::factory()->create();
        
        // ソフトデリート実行
        $task->delete();
        
        // データベースに物理的には残っていることを確認
        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
        ]);
        
        // deleted_atが設定されていることを確認
        $this->assertNotNull($task->fresh()->deleted_at);
        
        // 通常のクエリでは取得されないことを確認
        $this->assertNull(Task::find($task->id));
        
        // withTrashedを使用すると取得できることを確認
        $this->assertNotNull(Task::withTrashed()->find($task->id));
    }

    /**
     * バリデーションルールが適切に機能することをテスト（モデル内バリデーションがある場合）
     */
    public function test_task_model_validates_data_correctly(): void
    {
        $task = new Task();

        // title が必須であることをテスト
        $task->title = '';
        $task->priority = 'high';
        $task->user_id = User::factory()->create()->id;
        $task->category_id = Category::factory()->create()->id;

        // モデルレベルでのバリデーション（実装されている場合）
        try {
            $task->save();
            $this->fail('タイトルが空の場合、バリデーションエラーが発生すべきです');
        } catch (\Exception $e) {
            // バリデーションエラーが発生することを期待
            $this->assertTrue(true);
        }
    }
}