#!/bin/bash
#
# Laravel本番環境デプロイスクリプト
# 
# 使用方法: ./deploy.sh [branch]
# 例: ./deploy.sh main

# エラー時に処理を停止
set -e

# 設定
APP_DIR="/var/www/laravel"
BRANCH="${1:-main}"
BACKUP_DIR="/var/backups/laravel"
LOG_FILE="/var/log/laravel-deploy.log"
SLACK_WEBHOOK_URL="https://hooks.slack.com/services/YOUR/WEBHOOK/URL"

# カラー出力用の設定
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# ログ関数
log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# エラー処理
error_exit() {
    echo -e "${RED}エラー: $1${NC}" >&2
    log "ERROR: $1"
    
    # Slackに通知（オプション）
    if [ ! -z "$SLACK_WEBHOOK_URL" ]; then
        curl -X POST -H 'Content-type: application/json' \
            --data "{\"text\":\":x: デプロイ失敗: $1\"}" \
            "$SLACK_WEBHOOK_URL" 2>/dev/null
    fi
    
    exit 1
}

# 成功メッセージ
success_message() {
    echo -e "${GREEN}✓ $1${NC}"
    log "SUCCESS: $1"
}

# 警告メッセージ
warning_message() {
    echo -e "${YELLOW}⚠ $1${NC}"
    log "WARNING: $1"
}

# メンテナンスモードの確認
check_maintenance() {
    if [ -f "$APP_DIR/storage/framework/down" ]; then
        warning_message "アプリケーションは既にメンテナンスモードです"
        return 0
    fi
    return 1
}

# デプロイ開始
log "=========================================="
log "デプロイ開始: ブランチ $BRANCH"
log "=========================================="

# 1. ディレクトリ確認
if [ ! -d "$APP_DIR" ]; then
    error_exit "アプリケーションディレクトリが存在しません: $APP_DIR"
fi

cd "$APP_DIR" || error_exit "ディレクトリの移動に失敗しました"

# 2. 現在のバージョンをバックアップ
echo "バックアップを作成中..."
BACKUP_NAME="backup_$(date +%Y%m%d_%H%M%S)"
mkdir -p "$BACKUP_DIR"

# データベースのバックアップ
php artisan backup:run --only-db || warning_message "データベースバックアップに失敗しました"

# ファイルのバックアップ（.envファイルを含む）
tar -czf "$BACKUP_DIR/${BACKUP_NAME}_files.tar.gz" \
    --exclude="$APP_DIR/node_modules" \
    --exclude="$APP_DIR/vendor" \
    --exclude="$APP_DIR/storage/logs/*" \
    "$APP_DIR" 2>/dev/null || warning_message "ファイルバックアップに失敗しました"

success_message "バックアップ完了: $BACKUP_NAME"

# 3. メンテナンスモードを有効化
echo "メンテナンスモードを有効化中..."
php artisan down --message="システムメンテナンス中です。しばらくお待ちください。" \
    --retry=60 \
    --secret="YOUR_MAINTENANCE_SECRET"
success_message "メンテナンスモード有効化完了"

# 4. 最新のコードを取得
echo "最新のコードを取得中..."
git fetch origin || error_exit "Git fetchに失敗しました"
git checkout "$BRANCH" || error_exit "ブランチのチェックアウトに失敗しました"
git pull origin "$BRANCH" || error_exit "Git pullに失敗しました"
success_message "コード更新完了"

# 5. Composerの依存関係を更新
echo "Composer依存関係を更新中..."
composer install --no-dev --optimize-autoloader --no-interaction \
    || error_exit "Composer installに失敗しました"
success_message "Composer更新完了"

# 6. NPM依存関係を更新とビルド
echo "NPM依存関係を更新中..."
npm ci --production || warning_message "NPM ciに失敗しました"
npm run build || warning_message "NPMビルドに失敗しました"
success_message "フロントエンドビルド完了"

# 7. データベースマイグレーション
echo "データベースマイグレーションを実行中..."
php artisan migrate --force || {
    warning_message "マイグレーションに失敗しました。ロールバックを試みます..."
    php artisan migrate:rollback --force
    error_exit "マイグレーションエラー: ロールバックを実行しました"
}
success_message "マイグレーション完了"

# 8. キャッシュのクリアと再生成
echo "キャッシュを最適化中..."
php artisan cache:clear
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache
success_message "キャッシュ最適化完了"

# 9. キューワーカーの再起動
echo "キューワーカーを再起動中..."
php artisan queue:restart || warning_message "キューワーカーの再起動に失敗しました"
success_message "キューワーカー再起動完了"

# 10. パーミッションの設定
echo "パーミッションを設定中..."
chown -R www-data:www-data "$APP_DIR/storage" "$APP_DIR/bootstrap/cache"
chmod -R 775 "$APP_DIR/storage" "$APP_DIR/bootstrap/cache"
success_message "パーミッション設定完了"

# 11. OPcacheのリセット
echo "OPcacheをリセット中..."
if command -v cachetool &> /dev/null; then
    cachetool opcache:reset --fcgi=/var/run/php/php8.2-fpm.sock
    success_message "OPcacheリセット完了"
else
    warning_message "cachetoolが見つかりません。OPcacheのリセットをスキップします"
fi

# 12. ヘルスチェック
echo "ヘルスチェックを実行中..."
HEALTH_CHECK_URL="http://localhost/health"
HTTP_STATUS=$(curl -s -o /dev/null -w "%{http_code}" "$HEALTH_CHECK_URL")

if [ "$HTTP_STATUS" -eq 200 ]; then
    success_message "ヘルスチェック成功"
else
    error_exit "ヘルスチェック失敗: HTTPステータス $HTTP_STATUS"
fi

# 13. メンテナンスモードを解除
echo "メンテナンスモードを解除中..."
php artisan up
success_message "メンテナンスモード解除完了"

# 14. デプロイ完了通知
DEPLOY_TIME=$((SECONDS / 60))
log "=========================================="
log "デプロイ完了: ${DEPLOY_TIME}分"
log "=========================================="

echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${GREEN}✨ デプロイが正常に完了しました！${NC}"
echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Slackに成功通知（オプション）
if [ ! -z "$SLACK_WEBHOOK_URL" ]; then
    curl -X POST -H 'Content-type: application/json' \
        --data "{\"text\":\":white_check_mark: デプロイ成功 (${DEPLOY_TIME}分)\"}" \
        "$SLACK_WEBHOOK_URL" 2>/dev/null
fi

exit 0