<?php
/**
 * Laravel版: Post.php
 * app/Models/Post.php
 * 
 * Laravelの利点：
 * - Eloquent ORMによる直感的なデータ操作
 * - Mass Assignment保護
 * - 自動的なタイムスタンプ管理
 * - リレーション定義の簡潔さ
 */

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Post extends Model
{
    /**
     * Mass Assignment保護
     * 
     * $fillableで指定したフィールドのみ一括代入可能
     * これにより、悪意のあるユーザーが想定外のフィールドを
     * 更新することを防ぐ（例：is_adminフィールドなど）
     */
    protected $fillable = [
        'name',
        'message'
    ];
    
    /**
     * 日付として扱う属性
     * 
     * LaravelのCarbonライブラリで自動的に処理される
     * created_at, updated_atは自動的に含まれる
     */
    protected $dates = [
        'created_at',
        'updated_at'
    ];
    
    /**
     * アクセサの例：名前を常に適切にフォーマット
     */
    public function getFormattedNameAttribute()
    {
        return '投稿者: ' . $this->name;
    }
    
    /**
     * スコープの例：最新の投稿を取得
     */
    public function scopeRecent($query, $limit = 10)
    {
        return $query->orderBy('created_at', 'desc')->limit($limit);
    }
}