{{-- 
    resources/views/products/index.blade.php
    
    商品一覧ページ
    検索、フィルタリング、ページネーション機能付き
--}}

@extends('layouts.app')

@section('title', '商品一覧')

@section('breadcrumb')
    <li class="breadcrumb-item active" aria-current="page">商品一覧</li>
@endsection

@section('content')
    <div class="row">
        {{-- サイドバー（フィルター） --}}
        <div class="col-md-3">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">絞り込み検索</h5>
                </div>
                <div class="card-body">
                    <form method="GET" action="{{ route('products.index') }}">
                        {{-- 検索キーワード --}}
                        <div class="mb-3">
                            <label for="search" class="form-label">キーワード</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="search" 
                                   name="search" 
                                   value="{{ request('search') }}"
                                   placeholder="商品名・説明文">
                        </div>
                        
                        {{-- カテゴリー --}}
                        <div class="mb-3">
                            <label for="category" class="form-label">カテゴリー</label>
                            <select class="form-select" id="category" name="category">
                                <option value="">すべて</option>
                                @foreach($categories ?? [] as $category)
                                    <option value="{{ $category->id }}" 
                                            @if(request('category') == $category->id) selected @endif>
                                        {{ $category->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        
                        {{-- 価格帯 --}}
                        <div class="mb-3">
                            <label class="form-label">価格帯</label>
                            <div class="row g-2">
                                <div class="col">
                                    <input type="number" 
                                           class="form-control" 
                                           name="min_price" 
                                           value="{{ request('min_price') }}"
                                           placeholder="最小">
                                </div>
                                <div class="col-auto">〜</div>
                                <div class="col">
                                    <input type="number" 
                                           class="form-control" 
                                           name="max_price" 
                                           value="{{ request('max_price') }}"
                                           placeholder="最大">
                                </div>
                            </div>
                        </div>
                        
                        {{-- 在庫状況 --}}
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" 
                                       type="checkbox" 
                                       id="in_stock" 
                                       name="in_stock" 
                                       value="1"
                                       @if(request('in_stock')) checked @endif>
                                <label class="form-check-label" for="in_stock">
                                    在庫ありのみ表示
                                </label>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100">検索</button>
                        <a href="{{ route('products.index') }}" class="btn btn-outline-secondary w-100 mt-2">
                            リセット
                        </a>
                    </form>
                </div>
            </div>
        </div>
        
        {{-- メインコンテンツ --}}
        <div class="col-md-9">
            {{-- ヘッダー --}}
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1>商品一覧</h1>
                @auth
                    @if(auth()->user()->isAdmin())
                        <a href="{{ route('products.create') }}" class="btn btn-success">
                            新規商品追加
                        </a>
                    @endif
                @endauth
            </div>
            
            {{-- 検索結果情報 --}}
            @if($search)
                <div class="alert alert-info">
                    「{{ $search }}」の検索結果：{{ $products->total() }}件
                </div>
            @endif
            
            {{-- ソート --}}
            <div class="d-flex justify-content-end mb-3">
                <form method="GET" action="{{ route('products.index') }}" class="d-flex align-items-center">
                    {{-- 既存のパラメータを保持 --}}
                    @foreach(request()->except(['sort', 'order']) as $key => $value)
                        <input type="hidden" name="{{ $key }}" value="{{ $value }}">
                    @endforeach
                    
                    <label for="sort" class="me-2">並び替え：</label>
                    <select class="form-select form-select-sm w-auto" name="sort" id="sort" onchange="this.form.submit()">
                        <option value="created_at" @if(request('sort') == 'created_at') selected @endif>
                            新着順
                        </option>
                        <option value="price" @if(request('sort') == 'price') selected @endif>
                            価格順
                        </option>
                        <option value="name" @if(request('sort') == 'name') selected @endif>
                            名前順
                        </option>
                    </select>
                    
                    <select class="form-select form-select-sm w-auto ms-2" name="order" onchange="this.form.submit()">
                        <option value="asc" @if(request('order') == 'asc') selected @endif>昇順</option>
                        <option value="desc" @if(request('order') == 'desc') selected @endif>降順</option>
                    </select>
                </form>
            </div>
            
            {{-- 商品グリッド --}}
            @if($products->count() > 0)
                <div class="row g-4">
                    @foreach($products as $product)
                        <div class="col-md-4">
                            <div class="card h-100">
                                {{-- 商品画像 --}}
                                @if($product->image_path)
                                    <img src="{{ asset('storage/' . $product->image_path) }}" 
                                         class="card-img-top" 
                                         alt="{{ $product->name }}"
                                         style="height: 200px; object-fit: cover;">
                                @else
                                    <div class="card-img-top bg-light d-flex align-items-center justify-content-center" 
                                         style="height: 200px;">
                                        <span class="text-muted">No Image</span>
                                    </div>
                                @endif
                                
                                <div class="card-body d-flex flex-column">
                                    {{-- 商品名 --}}
                                    <h5 class="card-title">
                                        <a href="{{ route('products.show', $product) }}" class="text-decoration-none">
                                            {{ $product->name }}
                                        </a>
                                    </h5>
                                    
                                    {{-- カテゴリー --}}
                                    @if($product->category)
                                        <span class="badge bg-secondary mb-2">
                                            {{ $product->category->name }}
                                        </span>
                                    @endif
                                    
                                    {{-- 説明文（省略） --}}
                                    <p class="card-text flex-grow-1">
                                        {{ Str::limit($product->description, 100) }}
                                    </p>
                                    
                                    {{-- 価格と在庫 --}}
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <span class="h5 mb-0 text-primary">
                                            {{ $product->formatted_price }}
                                        </span>
                                        <span class="badge {{ $product->stock > 0 ? 'bg-success' : 'bg-danger' }}">
                                            {{ $product->availability_status }}
                                        </span>
                                    </div>
                                    
                                    {{-- アクションボタン --}}
                                    <div class="d-grid gap-2">
                                        <a href="{{ route('products.show', $product) }}" class="btn btn-primary">
                                            詳細を見る
                                        </a>
                                        @if($product->stock > 0)
                                            <button class="btn btn-outline-success add-to-cart" 
                                                    data-product-id="{{ $product->id }}">
                                                カートに追加
                                            </button>
                                        @else
                                            <button class="btn btn-outline-secondary" disabled>
                                                在庫なし
                                            </button>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
                
                {{-- ページネーション --}}
                <div class="mt-5">
                    {{ $products->appends(request()->query())->links('pagination::bootstrap-5') }}
                </div>
            @else
                {{-- 商品がない場合 --}}
                <div class="alert alert-warning text-center py-5">
                    <h4>商品が見つかりませんでした</h4>
                    <p>検索条件を変更して、もう一度お試しください。</p>
                    <a href="{{ route('products.index') }}" class="btn btn-primary mt-3">
                        すべての商品を表示
                    </a>
                </div>
            @endif
        </div>
    </div>
@endsection

@push('scripts')
<script>
    // カートに追加（Ajax処理の例）
    document.querySelectorAll('.add-to-cart').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.dataset.productId;
            
            fetch('/cart/add', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({
                    product_id: productId,
                    quantity: 1
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('カートに追加しました');
                } else {
                    alert('エラーが発生しました');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('エラーが発生しました');
            });
        });
    });
</script>
@endpush