# 第4章 サンプルコード - データベースとEloquent ORM

このディレクトリには、第4章で解説するデータベース操作とEloquent ORMのサンプルコードが含まれています。

## ディレクトリ構成

```
chapter04/
├── migrations/          # マイグレーションファイル
│   ├── create_users_table.php
│   ├── create_posts_table.php
│   ├── create_comments_table.php
│   └── create_tags_table.php
├── models/             # Eloquentモデル
│   ├── User.php
│   ├── Post.php
│   ├── Comment.php
│   └── Tag.php
├── seeders/            # シーダーファイル
│   ├── DatabaseSeeder.php
│   ├── UserSeeder.php
│   └── PostSeeder.php
└── factories/          # ファクトリーファイル
    ├── UserFactory.php
    └── PostFactory.php
```

## マイグレーション

### マイグレーションの作成

```bash
# テーブル作成用マイグレーション
php artisan make:migration create_posts_table

# テーブル変更用マイグレーション
php artisan make:migration add_published_at_to_posts_table --table=posts

# 外部キー追加用マイグレーション
php artisan make:migration add_foreign_keys_to_posts_table
```

### マイグレーションの実行

```bash
# マイグレーションを実行
php artisan migrate

# ロールバック（最後のバッチ）
php artisan migrate:rollback

# 全体をリセットして再実行
php artisan migrate:fresh

# シーダーも同時に実行
php artisan migrate:fresh --seed
```

## Eloquent ORM

### 基本的なCRUD操作

```php
// Create（作成）
$user = User::create([
    'name' => '山田太郎',
    'email' => 'yamada@example.com',
    'password' => bcrypt('password')
]);

// Read（読み取り）
$user = User::find(1);
$users = User::where('active', true)->get();
$user = User::where('email', 'yamada@example.com')->first();

// Update（更新）
$user->update(['name' => '山田次郎']);
User::where('active', false)->update(['deleted_at' => now()]);

// Delete（削除）
$user->delete();
User::where('created_at', '<', now()->subYear())->delete();
```

### リレーション

#### 1対多（One to Many）

```php
// User has many Posts
class User extends Model {
    public function posts() {
        return $this->hasMany(Post::class);
    }
}

// Post belongs to User
class Post extends Model {
    public function user() {
        return $this->belongsTo(User::class);
    }
}
```

#### 多対多（Many to Many）

```php
// Post has many Tags
class Post extends Model {
    public function tags() {
        return $this->belongsToMany(Tag::class);
    }
}

// Tag has many Posts
class Tag extends Model {
    public function posts() {
        return $this->belongsToMany(Post::class);
    }
}
```

## クエリビルダー

### 高度なクエリ

```php
// 複雑な条件
$posts = Post::where('status', 'published')
    ->where(function ($query) {
        $query->where('views', '>', 1000)
              ->orWhere('featured', true);
    })
    ->orderBy('created_at', 'desc')
    ->limit(10)
    ->get();

// 集計
$totalViews = Post::sum('views');
$avgRating = Post::avg('rating');
$postCount = Post::count();

// グループ化
$postsByCategory = Post::select('category_id', DB::raw('count(*) as total'))
    ->groupBy('category_id')
    ->having('total', '>', 5)
    ->get();
```

## シーダーとファクトリー

### シーダーの実行

```bash
# すべてのシーダーを実行
php artisan db:seed

# 特定のシーダーを実行
php artisan db:seed --class=UserSeeder

# マイグレーションと同時に実行
php artisan migrate:fresh --seed
```

### ファクトリーの使用

```php
// 1件作成
$user = User::factory()->create();

// 複数件作成
$users = User::factory()->count(10)->create();

// 関連データも同時に作成
$user = User::factory()
    ->has(Post::factory()->count(3))
    ->create();
```

## パフォーマンス最適化

### N+1問題の解決

```php
// 悪い例（N+1問題発生）
$posts = Post::all();
foreach ($posts as $post) {
    echo $post->user->name; // 各ループでクエリ発行
}

// 良い例（Eager Loading）
$posts = Post::with('user')->get();
foreach ($posts as $post) {
    echo $post->user->name; // 追加クエリなし
}
```

### クエリの最適化

```php
// インデックスの活用
Schema::table('posts', function (Blueprint $table) {
    $table->index('user_id');
    $table->index(['status', 'created_at']);
});

// チャンク処理
Post::chunk(100, function ($posts) {
    foreach ($posts as $post) {
        // 処理
    }
});
```

## トランザクション

```php
DB::beginTransaction();
try {
    $user = User::create($userData);
    $user->posts()->create($postData);
    
    DB::commit();
} catch (\Exception $e) {
    DB::rollBack();
    throw $e;
}
```

## 実行方法

1. データベースの設定（.env）

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=laravel_db
DB_USERNAME=root
DB_PASSWORD=password
```

2. マイグレーションの実行

```bash
php artisan migrate
```

3. シーダーの実行

```bash
php artisan db:seed
```

4. Tinkerで動作確認

```bash
php artisan tinker
>>> User::all()
>>> Post::with('user')->first()
```

## 学習のポイント

1. **マイグレーションによるバージョン管理**
   - データベース構造の変更履歴を管理
   - チーム開発での整合性確保

2. **Eloquent ORMの活用**
   - 直感的なデータ操作
   - リレーションの簡潔な表現

3. **パフォーマンスの意識**
   - N+1問題の理解と対策
   - 適切なインデックスの設計

4. **データ整合性の確保**
   - トランザクションの適切な使用
   - 外部キー制約の活用