<?php
/**
 * database/factories/UserFactory.php
 * 
 * ユーザーファクトリー
 * テスト用のダミーユーザーデータを生成
 */

namespace Database\Factories;

use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

class UserFactory extends Factory
{
    /**
     * モデル名
     */
    protected $model = User::class;
    
    /**
     * モデルのデフォルト状態を定義
     */
    public function definition(): array
    {
        return [
            'name' => $this->faker->name(),
            'email' => $this->faker->unique()->safeEmail(),
            'email_verified_at' => now(),
            'password' => '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', // password
            'remember_token' => Str::random(10),
            
            // プロフィール情報
            'avatar' => $this->faker->optional(0.3)->imageUrl(200, 200, 'people'),
            'bio' => $this->faker->optional(0.7)->paragraph(),
            'birth_date' => $this->faker->optional(0.5)->dateTimeBetween('-60 years', '-18 years'),
            'gender' => $this->faker->optional(0.6)->randomElement(['male', 'female', 'other']),
            
            // 連絡先情報
            'phone' => $this->faker->optional(0.4)->phoneNumber(),
            'address' => $this->faker->optional(0.3)->streetAddress(),
            'city' => $this->faker->optional(0.3)->city(),
            'postal_code' => $this->faker->optional(0.3)->postcode(),
            
            // ステータス
            'is_active' => $this->faker->boolean(90), // 90%の確率でアクティブ
            'is_admin' => false,
            'login_count' => $this->faker->numberBetween(0, 100),
            'last_login_at' => $this->faker->optional(0.8)->dateTimeBetween('-1 month', 'now'),
        ];
    }
    
    /**
     * メール未認証状態
     */
    public function unverified(): static
    {
        return $this->state(fn (array $attributes) => [
            'email_verified_at' => null,
        ]);
    }
    
    /**
     * 管理者状態
     */
    public function admin(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_admin' => true,
            'is_active' => true,
            'email_verified_at' => now(),
        ]);
    }
    
    /**
     * 非アクティブ状態
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
            'last_login_at' => null,
        ]);
    }
    
    /**
     * プロフィール完全入力状態
     */
    public function withCompleteProfile(): static
    {
        return $this->state(fn (array $attributes) => [
            'avatar' => $this->faker->imageUrl(200, 200, 'people'),
            'bio' => $this->faker->paragraph(3),
            'birth_date' => $this->faker->dateTimeBetween('-50 years', '-20 years'),
            'gender' => $this->faker->randomElement(['male', 'female', 'other']),
            'phone' => $this->faker->phoneNumber(),
            'address' => $this->faker->streetAddress(),
            'city' => $this->faker->city(),
            'postal_code' => $this->faker->postcode(),
        ]);
    }
    
    /**
     * 新規ユーザー状態（最近登録）
     */
    public function newUser(): static
    {
        return $this->state(fn (array $attributes) => [
            'created_at' => $this->faker->dateTimeBetween('-7 days', 'now'),
            'email_verified_at' => null,
            'login_count' => 0,
            'last_login_at' => null,
        ]);
    }
    
    /**
     * アクティブユーザー状態（頻繁にログイン）
     */
    public function activeUser(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
            'email_verified_at' => $this->faker->dateTimeBetween('-1 year', '-1 month'),
            'login_count' => $this->faker->numberBetween(50, 500),
            'last_login_at' => $this->faker->dateTimeBetween('-1 hour', 'now'),
        ]);
    }
    
    /**
     * 日本人ユーザー状態
     */
    public function japanese(): static
    {
        $faker = \Faker\Factory::create('ja_JP');
        
        return $this->state(fn (array $attributes) => [
            'name' => $faker->name(),
            'bio' => $faker->optional(0.7)->realText(200),
            'phone' => $faker->optional(0.4)->phoneNumber(),
            'address' => $faker->optional(0.3)->address(),
            'city' => $faker->optional(0.3)->city(),
            'postal_code' => $faker->optional(0.3)->postcode(),
        ]);
    }
}