<?php
/**
 * app/Models/Post.php
 * 
 * 投稿モデル
 * 複雑なリレーションとクエリスコープの実装例
 */

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Post extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * Mass Assignment可能な属性
     */
    protected $fillable = [
        'user_id',
        'title',
        'slug',
        'content',
        'excerpt',
        'featured_image',
        'meta_data',
        'status',
        'is_featured',
        'allow_comments',
        'published_at'
    ];
    
    /**
     * キャストする属性
     */
    protected $casts = [
        'meta_data' => 'array',
        'is_featured' => 'boolean',
        'allow_comments' => 'boolean',
        'view_count' => 'integer',
        'like_count' => 'integer',
        'comment_count' => 'integer',
        'published_at' => 'datetime',
    ];
    
    /**
     * 日付として扱う属性
     */
    protected $dates = [
        'published_at',
        'created_at',
        'updated_at',
        'deleted_at'
    ];
    
    /**
     * デフォルト値
     */
    protected $attributes = [
        'status' => 'draft',
        'is_featured' => false,
        'allow_comments' => true,
        'view_count' => 0,
        'like_count' => 0,
        'comment_count' => 0,
    ];
    
    /**
     * アクセサに追加する属性
     */
    protected $appends = [
        'reading_time',
        'is_published',
        'featured_image_url'
    ];
    
    /*
    |--------------------------------------------------------------------------
    | リレーション
    |--------------------------------------------------------------------------
    */
    
    /**
     * 投稿者（多対1）
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    /**
     * 著者（userのエイリアス）
     */
    public function author()
    {
        return $this->belongsTo(User::class, 'user_id');
    }
    
    /**
     * コメント（1対多）
     */
    public function comments()
    {
        return $this->hasMany(Comment::class);
    }
    
    /**
     * 承認済みコメント
     */
    public function approvedComments()
    {
        return $this->hasMany(Comment::class)->where('is_approved', true);
    }
    
    /**
     * タグ（多対多）
     */
    public function tags()
    {
        return $this->belongsToMany(Tag::class)
            ->withTimestamps();
    }
    
    /**
     * カテゴリー（多対1）
     */
    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    
    /**
     * いいねしたユーザー（多対多）
     */
    public function likedByUsers()
    {
        return $this->belongsToMany(User::class, 'likes')
            ->withTimestamps();
    }
    
    /**
     * 関連投稿（同じカテゴリーの投稿）
     */
    public function relatedPosts()
    {
        return $this->hasMany(Post::class, 'category_id', 'category_id')
            ->where('id', '!=', $this->id)
            ->published()
            ->limit(5);
    }
    
    /*
    |--------------------------------------------------------------------------
    | アクセサ（Getter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * 読了時間を計算（分）
     */
    public function getReadingTimeAttribute()
    {
        $wordsPerMinute = 200; // 平均的な読書速度
        $wordCount = str_word_count(strip_tags($this->content));
        $minutes = ceil($wordCount / $wordsPerMinute);
        
        return $minutes;
    }
    
    /**
     * 公開済みかチェック
     */
    public function getIsPublishedAttribute()
    {
        return $this->status === 'published' && 
               $this->published_at && 
               $this->published_at <= now();
    }
    
    /**
     * アイキャッチ画像のURLを取得
     */
    public function getFeaturedImageUrlAttribute()
    {
        if ($this->featured_image) {
            return asset('storage/' . $this->featured_image);
        }
        
        return asset('images/default-post.jpg');
    }
    
    /**
     * 記事のURLを取得
     */
    public function getUrlAttribute()
    {
        return route('posts.show', $this->slug);
    }
    
    /**
     * 短い説明文を取得
     */
    public function getShortDescriptionAttribute()
    {
        if ($this->excerpt) {
            return $this->excerpt;
        }
        
        return Str::limit(strip_tags($this->content), 150);
    }
    
    /*
    |--------------------------------------------------------------------------
    | ミューテータ（Setter）
    |--------------------------------------------------------------------------
    */
    
    /**
     * タイトル設定時にスラッグを自動生成
     */
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = $value;
        
        // スラッグが空の場合のみ自動生成
        if (empty($this->attributes['slug'])) {
            $this->attributes['slug'] = $this->generateUniqueSlug($value);
        }
    }
    
    /**
     * ユニークなスラッグを生成
     */
    protected function generateUniqueSlug($title)
    {
        $slug = Str::slug($title);
        $originalSlug = $slug;
        $count = 1;
        
        while (static::where('slug', $slug)->exists()) {
            $slug = "{$originalSlug}-{$count}";
            $count++;
        }
        
        return $slug;
    }
    
    /*
    |--------------------------------------------------------------------------
    | スコープ
    |--------------------------------------------------------------------------
    */
    
    /**
     * 公開済みの投稿のみ
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
                     ->where('published_at', '<=', now());
    }
    
    /**
     * 下書きのみ
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }
    
    /**
     * おすすめ投稿のみ
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
    
    /**
     * 人気順（閲覧数順）
     */
    public function scopePopular($query)
    {
        return $query->orderBy('view_count', 'desc');
    }
    
    /**
     * トレンド（最近の人気投稿）
     */
    public function scopeTrending($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days))
                     ->orderBy('view_count', 'desc')
                     ->orderBy('like_count', 'desc');
    }
    
    /**
     * カテゴリーで絞り込み
     */
    public function scopeInCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }
    
    /**
     * タグで絞り込み
     */
    public function scopeWithTag($query, $tagId)
    {
        return $query->whereHas('tags', function ($q) use ($tagId) {
            $q->where('tags.id', $tagId);
        });
    }
    
    /**
     * 著者で絞り込み
     */
    public function scopeByAuthor($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
    
    /**
     * 検索スコープ
     */
    public function scopeSearch($query, $keyword)
    {
        return $query->where(function ($q) use ($keyword) {
            $q->where('title', 'like', "%{$keyword}%")
              ->orWhere('content', 'like', "%{$keyword}%")
              ->orWhere('excerpt', 'like', "%{$keyword}%");
        });
    }
    
    /**
     * 期間で絞り込み
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('published_at', [$startDate, $endDate]);
    }
    
    /*
    |--------------------------------------------------------------------------
    | ビジネスロジック
    |--------------------------------------------------------------------------
    */
    
    /**
     * 投稿を公開
     */
    public function publish()
    {
        $this->update([
            'status' => 'published',
            'published_at' => $this->published_at ?? now()
        ]);
    }
    
    /**
     * 下書きに戻す
     */
    public function unpublish()
    {
        $this->update(['status' => 'draft']);
    }
    
    /**
     * 閲覧数を増やす
     */
    public function incrementViewCount()
    {
        $this->increment('view_count');
    }
    
    /**
     * タグを同期
     */
    public function syncTags(array $tagIds)
    {
        $this->tags()->sync($tagIds);
    }
    
    /**
     * タグを名前から同期
     */
    public function syncTagsByNames(array $tagNames)
    {
        $tagIds = [];
        
        foreach ($tagNames as $name) {
            $tag = Tag::firstOrCreate(['name' => $name]);
            $tagIds[] = $tag->id;
        }
        
        $this->tags()->sync($tagIds);
    }
    
    /**
     * コメントを追加
     */
    public function addComment($data)
    {
        return $this->comments()->create($data);
    }
    
    /**
     * 統計情報を取得
     */
    public function getStats()
    {
        return [
            'views' => $this->view_count,
            'likes' => $this->like_count,
            'comments' => $this->comment_count,
            'reading_time' => $this->reading_time,
        ];
    }
    
    /*
    |--------------------------------------------------------------------------
    | イベント
    |--------------------------------------------------------------------------
    */
    
    /**
     * モデルイベントの登録
     */
    protected static function booted()
    {
        // 作成時の処理
        static::creating(function ($post) {
            // ユーザーIDが設定されていない場合は現在のユーザーを設定
            if (empty($post->user_id)) {
                $post->user_id = auth()->id();
            }
        });
        
        // 削除時の処理
        static::deleting(function ($post) {
            // ソフトデリートの場合は処理しない
            if (!$post->isForceDeleting()) {
                return;
            }
            
            // アイキャッチ画像を削除
            if ($post->featured_image) {
                \Storage::disk('public')->delete($post->featured_image);
            }
            
            // 関連データを削除
            $post->comments()->delete();
            $post->tags()->detach();
        });
    }
}