<?php
/**
 * database/seeders/DatabaseSeeder.php
 * 
 * メインシーダー
 * 他のシーダーを呼び出してデータベースに初期データを投入
 */

namespace Database\Seeders;

use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    /**
     * データベースシーディングを実行
     */
    public function run(): void
    {
        // 外部キー制約を一時的に無効化（開発環境のみ）
        \DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // 既存データをクリア
        $this->truncateTables();
        
        // 外部キー制約を再度有効化
        \DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        // シーダーを実行
        $this->call([
            UserSeeder::class,       // ユーザーデータ
            CategorySeeder::class,   // カテゴリーデータ
            TagSeeder::class,        // タグデータ
            PostSeeder::class,       // 投稿データ
            CommentSeeder::class,    // コメントデータ
        ]);
        
        // 管理者ユーザーを作成
        $this->createAdminUser();
        
        // 統計情報を出力
        $this->command->info('Seeding completed successfully!');
        $this->displayStatistics();
    }
    
    /**
     * テーブルをクリア
     */
    private function truncateTables(): void
    {
        // 削除順序に注意（外部キー制約のため）
        \App\Models\Comment::truncate();
        \App\Models\Post::truncate();
        \App\Models\Tag::truncate();
        \App\Models\Category::truncate();
        \App\Models\User::truncate();
        
        // 中間テーブルもクリア
        \DB::table('post_tag')->truncate();
        \DB::table('likes')->truncate();
        \DB::table('follows')->truncate();
    }
    
    /**
     * 管理者ユーザーを作成
     */
    private function createAdminUser(): void
    {
        \App\Models\User::create([
            'name' => '管理者',
            'email' => 'admin@example.com',
            'password' => 'password', // ミューテータでハッシュ化される
            'email_verified_at' => now(),
            'is_admin' => true,
            'is_active' => true,
            'bio' => 'システム管理者アカウント',
        ]);
        
        $this->command->info('Admin user created: admin@example.com / password');
    }
    
    /**
     * シーディング結果の統計を表示
     */
    private function displayStatistics(): void
    {
        $stats = [
            'Users' => \App\Models\User::count(),
            'Posts' => \App\Models\Post::count(),
            'Comments' => \App\Models\Comment::count(),
            'Categories' => \App\Models\Category::count(),
            'Tags' => \App\Models\Tag::count(),
        ];
        
        $this->command->table(['Model', 'Count'], array_map(
            fn($model, $count) => [$model, $count],
            array_keys($stats),
            array_values($stats)
        ));
    }
}