<?php

/**
 * タスクテーブル作成マイグレーション
 * 
 * 第5章「実用的なCRUDアプリケーション開発」サンプルコード
 * 
 * 機能:
 * - タスク管理システムの中核テーブル
 * - 外部キー制約とインデックス最適化
 * - ソフトデリート対応
 * - 全文検索対応
 * - JSON カラムによる柔軟なメタデータ保存
 */

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * マイグレーション実行
     *
     * @return void
     */
    public function up(): void
    {
        Schema::create('tasks', function (Blueprint $table) {
            // 基本ID・タイムスタンプ
            $table->id();
            $table->timestamps();
            $table->softDeletes(); // ソフトデリート対応
            
            // 基本情報
            $table->string('title', 255)->comment('タスクタイトル');
            $table->text('description')->nullable()->comment('タスク説明');
            
            // ステータス・優先度
            $table->enum('status', ['pending', 'in_progress', 'completed', 'archived', 'cancelled'])
                  ->default('pending')
                  ->comment('タスクステータス');
            $table->tinyInteger('priority')
                  ->unsigned()
                  ->default(2)
                  ->comment('優先度 1:低 2:中 3:高 4:緊急');
            
            // 進捗・工数管理
            $table->tinyInteger('progress')
                  ->unsigned()
                  ->default(0)
                  ->comment('進捗率 0-100%');
            $table->decimal('estimated_hours', 6, 2)
                  ->nullable()
                  ->comment('予定工数（時間）');
            $table->decimal('actual_hours', 6, 2)
                  ->nullable()
                  ->comment('実績工数（時間）');
            
            // 日付関連
            $table->date('due_date')
                  ->nullable()
                  ->comment('期限日');
            $table->timestamp('completed_at')
                  ->nullable()
                  ->comment('完了日時');
            $table->timestamp('started_at')
                  ->nullable()
                  ->comment('開始日時');
            
            // 関連情報
            $table->foreignId('user_id')
                  ->constrained('users')
                  ->onUpdate('cascade')
                  ->onDelete('cascade')
                  ->comment('担当ユーザーID');
            $table->foreignId('category_id')
                  ->constrained('categories')
                  ->onUpdate('cascade')
                  ->onDelete('restrict')
                  ->comment('カテゴリID');
            $table->foreignId('parent_id')
                  ->nullable()
                  ->constrained('tasks')
                  ->onUpdate('cascade')
                  ->onDelete('set null')
                  ->comment('親タスクID（サブタスク用）');
            
            // JSON カラム（柔軟なメタデータ保存）
            $table->json('attachments')
                  ->nullable()
                  ->comment('添付ファイル情報（JSON）');
            $table->json('metadata')
                  ->nullable()
                  ->comment('追加メタデータ（JSON）');
            
            // 追加フィールド
            $table->text('completion_note')
                  ->nullable()
                  ->comment('完了時の備考');
            $table->text('reason')
                  ->nullable()
                  ->comment('高優先度の理由');
            $table->boolean('is_template')
                  ->default(false)
                  ->comment('テンプレートフラグ');
            $table->boolean('is_recurring')
                  ->default(false)
                  ->comment('繰り返しタスクフラグ');
            $table->string('recurring_pattern')
                  ->nullable()
                  ->comment('繰り返しパターン');
            
            // インデックス作成（パフォーマンス最適化）
            $table->index(['user_id', 'status']); // ユーザー別ステータス検索
            $table->index(['user_id', 'priority']); // ユーザー別優先度検索  
            $table->index(['status', 'priority']); // ステータス・優先度複合検索
            $table->index(['due_date', 'status']); // 期限日・ステータス検索
            $table->index('category_id'); // カテゴリ検索
            $table->index('completed_at'); // 完了日検索
            $table->index(['created_at', 'user_id']); // 作成日・ユーザー検索
            $table->index('parent_id'); // 親タスク検索（サブタスク用）
            
            // 全文検索インデックス（MySQL 5.7以降）
            if (config('database.default') === 'mysql') {
                $table->fullText(['title', 'description'], 'tasks_fulltext_index');
            }
        });
        
        // テーブル作成後の追加設定
        if (config('database.default') === 'mysql') {
            // テーブルエンジン・文字セット設定
            DB::statement('ALTER TABLE tasks ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci');
            
            // テーブルコメント設定
            DB::statement("ALTER TABLE tasks COMMENT='タスク管理テーブル'");
        }
    }

    /**
     * マイグレーション巻き戻し
     *
     * @return void
     */
    public function down(): void
    {
        // 外部キー制約を持つテーブルなので、制約を確認してから削除
        Schema::table('tasks', function (Blueprint $table) {
            // 外部キー制約を明示的に削除
            $table->dropForeign(['user_id']);
            $table->dropForeign(['category_id']);
            $table->dropForeign(['parent_id']);
        });
        
        Schema::dropIfExists('tasks');
    }
};