<?php

/**
 * Taskモデル - タスク管理システムの中核モデル
 * 
 * 第5章「実用的なCRUDアプリケーション開発」サンプルコード
 * 
 * 機能:
 * - Eloquent リレーション定義
 * - クエリスコープによる検索機能
 * - アクセサ・ミューテータによるデータ操作
 * - ソフトデリート対応
 * - ビジネスロジックメソッド
 */

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Carbon\Carbon;

class Task extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * Mass Assignment で設定可能な属性
     * 
     * @var array<string>
     */
    protected $fillable = [
        'title',
        'description',
        'status',
        'priority',
        'due_date',
        'completed_at',
        'user_id',
        'category_id',
        'attachments',
        'tags',
        'progress',
        'estimated_hours',
        'actual_hours',
    ];

    /**
     * 属性のキャスト定義
     * 
     * @var array<string, string>
     */
    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'priority' => 'integer',
        'progress' => 'integer',
        'estimated_hours' => 'decimal:2',
        'actual_hours' => 'decimal:2',
        'attachments' => 'array',
        'tags' => 'array',
    ];

    /**
     * デフォルト値
     * 
     * @var array<string, mixed>
     */
    protected $attributes = [
        'status' => 'pending',
        'priority' => 2,
        'progress' => 0,
    ];

    /*
    |--------------------------------------------------------------------------
    | リレーション定義
    |--------------------------------------------------------------------------
    */

    /**
     * タスクを所有するユーザー（1対多の逆）
     * 
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * タスクが所属するカテゴリ（1対多の逆）
     * 
     * @return BelongsTo
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * タスクに関連するタグ（多対多）
     * 
     * @return BelongsToMany
     */
    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(Tag::class, 'task_tag')
                    ->withTimestamps();
    }

    /**
     * タスクに付いたコメント（ポリモーフィック1対多）
     * 
     * @return MorphMany
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Comment::class, 'commentable')
                    ->latest();
    }

    /**
     * タスクの添付ファイル（1対多）
     * 
     * @return HasMany
     */
    public function attachments(): HasMany
    {
        return $this->hasMany(TaskAttachment::class);
    }

    /*
    |--------------------------------------------------------------------------
    | クエリスコープ（検索・フィルタリング機能）
    |--------------------------------------------------------------------------
    */

    /**
     * ステータスによるフィルタ
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * 未着手タスクのフィルタ
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * 進行中タスクのフィルタ
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    /**
     * 完了タスクのフィルタ
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * 期限切れタスクのフィルタ
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOverdue($query)
    {
        return $query->where('due_date', '<', now())
                    ->whereNotIn('status', ['completed', 'archived']);
    }

    /**
     * 現在のユーザーのタスクのみ取得
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeMyTasks($query)
    {
        return $query->where('user_id', auth()->id());
    }

    /**
     * 優先度による絞り込み
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $priority
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    /**
     * 高優先度タスクの取得
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeHighPriority($query)
    {
        return $query->where('priority', '>=', 3);
    }

    /**
     * 期限日でソート（昇順）
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOrderByDueDate($query)
    {
        return $query->orderBy('due_date', 'asc');
    }

    /**
     * 優先度でソート（降順）
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOrderByPriority($query)
    {
        return $query->orderBy('priority', 'desc');
    }

    /**
     * 検索機能（タイトル・説明文）
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $search
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, $search)
    {
        return $query->where('title', 'LIKE', "%{$search}%")
                    ->orWhere('description', 'LIKE', "%{$search}%");
    }

    /*
    |--------------------------------------------------------------------------
    | アクセサ（値の取得時の変換）
    |--------------------------------------------------------------------------
    */

    /**
     * ステータス表示名を取得
     * 
     * @return string
     */
    public function getStatusDisplayAttribute(): string
    {
        return self::getStatusOptions()[$this->status] ?? $this->status;
    }

    /**
     * 優先度表示名を取得
     * 
     * @return string
     */
    public function getPriorityDisplayAttribute(): string
    {
        return self::getPriorityOptions()[$this->priority] ?? $this->priority;
    }

    /**
     * 期限まで残り日数を取得
     * 
     * @return int|null
     */
    public function getDaysUntilDueAttribute(): ?int
    {
        if (!$this->due_date) {
            return null;
        }

        return now()->diffInDays($this->due_date, false);
    }

    /**
     * 期限切れかどうか判定
     * 
     * @return bool
     */
    public function getIsOverdueAttribute(): bool
    {
        if (!$this->due_date || $this->status === 'completed') {
            return false;
        }

        return $this->due_date->isPast();
    }

    /**
     * 進捗パーセンテージを取得
     * 
     * @return int
     */
    public function getProgressPercentageAttribute(): int
    {
        return min(100, max(0, $this->progress));
    }

    /*
    |--------------------------------------------------------------------------
    | ミューテータ（値の設定時の変換）
    |--------------------------------------------------------------------------
    */

    /**
     * タイトル設定時の自動トリム
     * 
     * @param string $value
     */
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = trim($value);
    }

    /**
     * 説明文設定時の自動トリム
     * 
     * @param string|null $value
     */
    public function setDescriptionAttribute($value)
    {
        $this->attributes['description'] = $value ? trim($value) : null;
    }

    /*
    |--------------------------------------------------------------------------
    | 静的メソッド（選択肢・定数）
    |--------------------------------------------------------------------------
    */

    /**
     * ステータスの選択肢を取得
     * 
     * @return array<string, string>
     */
    public static function getStatusOptions(): array
    {
        return [
            'pending' => '未着手',
            'in_progress' => '進行中',
            'completed' => '完了',
            'archived' => 'アーカイブ済み',
            'cancelled' => 'キャンセル',
        ];
    }

    /**
     * 優先度の選択肢を取得
     * 
     * @return array<int, string>
     */
    public static function getPriorityOptions(): array
    {
        return [
            1 => '低',
            2 => '中',
            3 => '高',
            4 => '緊急',
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | ビジネスロジックメソッド
    |--------------------------------------------------------------------------
    */

    /**
     * タスクを完了状態にする
     * 
     * @return void
     */
    public function markAsCompleted(): void
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
            'progress' => 100,
        ]);
    }

    /**
     * タスクを進行中状態にする
     * 
     * @return void
     */
    public function markAsInProgress(): void
    {
        $this->update([
            'status' => 'in_progress',
        ]);
    }

    /**
     * 進捗率を更新
     * 
     * @param int $progress
     * @return void
     */
    public function updateProgress(int $progress): void
    {
        $progress = min(100, max(0, $progress));
        
        $this->update(['progress' => $progress]);
        
        // 進捗率100%の場合は自動的に完了状態にする
        if ($progress === 100 && $this->status !== 'completed') {
            $this->markAsCompleted();
        }
    }

    /**
     * 実績時間を追加
     * 
     * @param float $hours
     * @return void
     */
    public function addActualHours(float $hours): void
    {
        $this->increment('actual_hours', $hours);
    }

    /**
     * タスクの詳細統計情報を取得
     * 
     * @return array
     */
    public function getDetailedStats(): array
    {
        return [
            'comments_count' => $this->comments()->count(),
            'attachments_count' => is_array($this->attachments) ? count($this->attachments) : 0,
            'tags_count' => $this->tags()->count(),
            'days_until_due' => $this->days_until_due,
            'is_overdue' => $this->is_overdue,
            'progress_percentage' => $this->progress_percentage,
            'estimated_vs_actual' => [
                'estimated' => $this->estimated_hours,
                'actual' => $this->actual_hours,
                'variance' => $this->actual_hours - $this->estimated_hours,
            ],
        ];
    }

    /**
     * タスクが編集可能かどうか判定
     * 
     * @return bool
     */
    public function canEdit(): bool
    {
        // 完了やアーカイブ済みのタスクは編集不可
        return !in_array($this->status, ['completed', 'archived']);
    }

    /**
     * タスクが削除可能かどうか判定
     * 
     * @return bool
     */
    public function canDelete(): bool
    {
        // 自分が作成したタスク、または管理者のみ削除可能
        return $this->user_id === auth()->id() || auth()->user()->isAdmin();
    }

    /*
    |--------------------------------------------------------------------------
    | イベントハンドラ
    |--------------------------------------------------------------------------
    */

    /**
     * モデル起動時の処理
     * 
     * @return void
     */
    protected static function boot()
    {
        parent::boot();

        // 作成時にデフォルト値を設定
        static::creating(function ($task) {
            if (!$task->user_id) {
                $task->user_id = auth()->id();
            }
        });

        // 更新時の処理
        static::updating(function ($task) {
            // ステータスが完了に変更された場合
            if ($task->isDirty('status') && $task->status === 'completed') {
                $task->completed_at = now();
                $task->progress = 100;
            }
        });
    }
}