<?php

/**
 * Web Routes - タスク管理システム
 * 
 * 第5章「実用的なCRUDアプリケーション開発」サンプルコード
 * 
 * RESTfulルーティングの実装例
 * - CRUD操作の標準的なルート設計
 * - ミドルウェアによる認証・認可
 * - ルートモデルバインディング
 * - 追加機能のカスタムルート
 */

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\TagController;
use App\Http\Controllers\DashboardController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

// ホームページ - ログイン状況に応じてリダイレクト
Route::get('/', function () {
    if (auth()->check()) {
        return redirect()->route('tasks.index');
    }
    return redirect()->route('login');
})->name('home');

// 認証が不要な公開ルート
Route::get('/about', function () {
    return view('pages.about');
})->name('about');

Route::get('/contact', function () {
    return view('pages.contact');
})->name('contact');

/*
|--------------------------------------------------------------------------
| 認証が必要なルート群
|--------------------------------------------------------------------------
*/

Route::middleware(['auth', 'verified'])->group(function () {
    
    // ダッシュボード
    Route::get('/dashboard', [DashboardController::class, 'index'])
        ->name('dashboard');

    /*
    |--------------------------------------------------------------------------
    | タスク管理（メインCRUDリソース）
    |--------------------------------------------------------------------------
    */
    
    // 基本的なCRUDルート（リソースルート）
    Route::resource('tasks', TaskController::class);
    
    // タスクの追加機能ルート
    Route::prefix('tasks')->name('tasks.')->group(function () {
        
        // 統計・レポート機能
        Route::get('statistics', [TaskController::class, 'statistics'])
            ->name('statistics');
            
        Route::get('reports', [TaskController::class, 'reports'])
            ->name('reports');
            
        Route::get('export', [TaskController::class, 'export'])
            ->name('export');
        
        // 検索・フィルタリング
        Route::get('search', [TaskController::class, 'search'])
            ->name('search');
            
        Route::get('filter', [TaskController::class, 'filter'])
            ->name('filter');
        
        // 一括操作
        Route::post('bulk-action', [TaskController::class, 'bulkAction'])
            ->name('bulk-action');
            
        Route::post('bulk-update', [TaskController::class, 'bulkUpdate'])
            ->name('bulk-update');
            
        Route::delete('bulk-delete', [TaskController::class, 'bulkDelete'])
            ->name('bulk-delete');
        
        // タスクのステータス変更（Ajax対応）
        Route::patch('{task}/status', [TaskController::class, 'updateStatus'])
            ->name('update-status');
            
        Route::patch('{task}/priority', [TaskController::class, 'updatePriority'])
            ->name('update-priority');
            
        Route::patch('{task}/progress', [TaskController::class, 'updateProgress'])
            ->name('update-progress');
        
        // タスクの複製・テンプレート機能
        Route::post('{task}/duplicate', [TaskController::class, 'duplicate'])
            ->name('duplicate');
            
        Route::post('{task}/template', [TaskController::class, 'saveAsTemplate'])
            ->name('save-template');
        
        // ファイル関連
        Route::post('{task}/attachments', [TaskController::class, 'addAttachment'])
            ->name('add-attachment');
            
        Route::delete('{task}/attachments/{attachment}', [TaskController::class, 'removeAttachment'])
            ->name('remove-attachment');
            
        Route::get('{task}/attachments/{attachment}/download', [TaskController::class, 'downloadAttachment'])
            ->name('download-attachment');
        
        // コメント機能（ネストしたリソース）
        Route::prefix('{task}')->group(function () {
            Route::resource('comments', CommentController::class)
                ->except(['index', 'show'])
                ->names([
                    'create' => 'comments.create',
                    'store' => 'comments.store',
                    'edit' => 'comments.edit',
                    'update' => 'comments.update',
                    'destroy' => 'comments.destroy',
                ]);
        });
        
        // タスク履歴・活動ログ
        Route::get('{task}/history', [TaskController::class, 'history'])
            ->name('history');
            
        Route::get('{task}/activities', [TaskController::class, 'activities'])
            ->name('activities');
        
        // ソフトデリート関連
        Route::get('trash', [TaskController::class, 'trash'])
            ->name('trash');
            
        Route::patch('{task}/restore', [TaskController::class, 'restore'])
            ->name('restore')
            ->withTrashed();
            
        Route::delete('{task}/force-delete', [TaskController::class, 'forceDelete'])
            ->name('force-delete')
            ->withTrashed();
    });

    /*
    |--------------------------------------------------------------------------
    | カテゴリ管理
    |--------------------------------------------------------------------------
    */
    
    Route::resource('categories', CategoryController::class);
    
    Route::prefix('categories')->name('categories.')->group(function () {
        // カテゴリの並び順変更
        Route::post('reorder', [CategoryController::class, 'reorder'])
            ->name('reorder');
            
        // カテゴリ内タスク一覧
        Route::get('{category}/tasks', [CategoryController::class, 'tasks'])
            ->name('tasks');
            
        // カテゴリ統計
        Route::get('{category}/statistics', [CategoryController::class, 'statistics'])
            ->name('statistics');
    });

    /*
    |--------------------------------------------------------------------------
    | タグ管理
    |--------------------------------------------------------------------------
    */
    
    Route::resource('tags', TagController::class)
        ->except(['show']);
    
    Route::prefix('tags')->name('tags.')->group(function () {
        // タグ検索（Ajax）
        Route::get('search', [TagController::class, 'search'])
            ->name('search');
            
        // タグ別タスク一覧
        Route::get('{tag}/tasks', [TagController::class, 'tasks'])
            ->name('tasks');
            
        // 人気タグ一覧
        Route::get('popular', [TagController::class, 'popular'])
            ->name('popular');
    });

    /*
    |--------------------------------------------------------------------------
    | API風ルート（Ajax用）
    |--------------------------------------------------------------------------
    */
    
    Route::prefix('api')->name('api.')->group(function () {
        // タスクの簡単な操作（Ajax専用）
        Route::get('tasks/count', [TaskController::class, 'count'])
            ->name('tasks.count');
            
        Route::get('tasks/{task}/quick-info', [TaskController::class, 'quickInfo'])
            ->name('tasks.quick-info');
            
        Route::patch('tasks/{task}/toggle-complete', [TaskController::class, 'toggleComplete'])
            ->name('tasks.toggle-complete');
        
        // 候補データ取得（オートコンプリート用）
        Route::get('categories/suggestions', [CategoryController::class, 'suggestions'])
            ->name('categories.suggestions');
            
        Route::get('tags/suggestions', [TagController::class, 'suggestions'])
            ->name('tags.suggestions');
            
        Route::get('users/suggestions', [UserController::class, 'suggestions'])
            ->name('users.suggestions');
        
        // 統計データ（チャート用）
        Route::get('stats/tasks-by-status', [TaskController::class, 'statsByStatus'])
            ->name('stats.tasks-by-status');
            
        Route::get('stats/tasks-by-priority', [TaskController::class, 'statsByPriority'])
            ->name('stats.tasks-by-priority');
            
        Route::get('stats/tasks-by-date', [TaskController::class, 'statsByDate'])
            ->name('stats.tasks-by-date');
    });

    /*
    |--------------------------------------------------------------------------
    | ユーザープロファイル・設定
    |--------------------------------------------------------------------------
    */
    
    Route::prefix('profile')->name('profile.')->group(function () {
        Route::get('/', [ProfileController::class, 'show'])
            ->name('show');
            
        Route::get('edit', [ProfileController::class, 'edit'])
            ->name('edit');
            
        Route::patch('/', [ProfileController::class, 'update'])
            ->name('update');
            
        Route::delete('/', [ProfileController::class, 'destroy'])
            ->name('destroy');
        
        // ユーザーのタスク統計
        Route::get('statistics', [ProfileController::class, 'statistics'])
            ->name('statistics');
            
        // 通知設定
        Route::get('notifications', [ProfileController::class, 'notifications'])
            ->name('notifications');
            
        Route::patch('notifications', [ProfileController::class, 'updateNotifications'])
            ->name('update-notifications');
    });

    /*
    |--------------------------------------------------------------------------
    | 管理者専用ルート
    |--------------------------------------------------------------------------
    */
    
    Route::middleware(['admin'])->prefix('admin')->name('admin.')->group(function () {
        // 管理ダッシュボード
        Route::get('/', [AdminController::class, 'dashboard'])
            ->name('dashboard');
        
        // システム統計
        Route::get('statistics', [AdminController::class, 'statistics'])
            ->name('statistics');
            
        // ユーザー管理
        Route::resource('users', UserController::class);
        
        // システムログ
        Route::get('logs', [AdminController::class, 'logs'])
            ->name('logs');
            
        // システム設定
        Route::get('settings', [AdminController::class, 'settings'])
            ->name('settings');
            
        Route::patch('settings', [AdminController::class, 'updateSettings'])
            ->name('update-settings');
    });

    /*
    |--------------------------------------------------------------------------
    | その他の機能ルート
    |--------------------------------------------------------------------------
    */
    
    // カレンダー表示
    Route::get('calendar', [CalendarController::class, 'index'])
        ->name('calendar');
        
    Route::get('calendar/events', [CalendarController::class, 'events'])
        ->name('calendar.events');
    
    // 通知機能
    Route::get('notifications', [NotificationController::class, 'index'])
        ->name('notifications.index');
        
    Route::patch('notifications/{notification}/read', [NotificationController::class, 'markAsRead'])
        ->name('notifications.read');
        
    Route::patch('notifications/read-all', [NotificationController::class, 'markAllAsRead'])
        ->name('notifications.read-all');
    
    // ヘルプ・サポート
    Route::get('help', [HelpController::class, 'index'])
        ->name('help');
        
    Route::get('help/{topic}', [HelpController::class, 'topic'])
        ->name('help.topic');
        
    Route::post('support/contact', [HelpController::class, 'contact'])
        ->name('support.contact');
});

/*
|--------------------------------------------------------------------------
| ルートモデルバインディングの設定例
|--------------------------------------------------------------------------
*/

// カスタムルートモデルバインディング
Route::bind('task', function ($value) {
    return \App\Models\Task::where('id', $value)
        ->where('user_id', auth()->id()) // ユーザーのタスクのみ
        ->firstOrFail();
});

// ソフトデリートを含むバインディング
Route::bind('task_with_trashed', function ($value) {
    return \App\Models\Task::withTrashed()
        ->where('id', $value)
        ->where('user_id', auth()->id())
        ->firstOrFail();
});

/*
|--------------------------------------------------------------------------
| フォールバックルート（404対応）
|--------------------------------------------------------------------------
*/

// 404ページのカスタマイズ
Route::fallback(function () {
    if (request()->expectsJson()) {
        return response()->json([
            'message' => 'ページが見つかりません。'
        ], 404);
    }
    
    return response()->view('errors.404', [], 404);
});

/*
|--------------------------------------------------------------------------
| 開発・デバッグ用ルート（本番環境では無効化）
|--------------------------------------------------------------------------
*/

if (app()->environment('local', 'staging')) {
    // デバッグ用ルート
    Route::get('/debug/tasks', function () {
        return \App\Models\Task::with(['user', 'category', 'tags'])
            ->latest()
            ->take(10)
            ->get()
            ->toArray();
    })->name('debug.tasks');
    
    // テストデータ生成
    Route::get('/debug/seed-tasks', function () {
        \App\Models\Task::factory()->count(50)->create([
            'user_id' => auth()->id()
        ]);
        return redirect()->route('tasks.index')
            ->with('success', 'テストデータを50件作成しました。');
    })->middleware('auth')->name('debug.seed');
}