@extends('layouts.app')

@section('title', 'タスク一覧')

@section('header')
    <div class="d-flex w-100">
        <div>
            <h2 class="mb-0">
                <i class="fas fa-tasks text-primary me-2"></i>タスク一覧
            </h2>
            <p class="text-muted mb-0">
                全{{ $tasks->total() }}件のタスク
                @if(request('search'))
                    （「{{ request('search') }}」の検索結果）
                @endif
            </p>
        </div>
        <div class="ms-auto">
            <a href="{{ route('tasks.create') }}" class="btn btn-primary">
                <i class="fas fa-plus me-1"></i>新規タスク作成
            </a>
        </div>
    </div>
@endsection

@section('content')
<div x-data="taskIndex()">
    <!-- フィルター・検索バー -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" action="{{ route('tasks.index') }}" x-ref="filterForm">
                <div class="row g-3 align-items-end">
                    <!-- 検索キーワード -->
                    <div class="col-md-3">
                        <label class="form-label">検索キーワード</label>
                        <input type="text" class="form-control" name="search" 
                               value="{{ request('search') }}"
                               placeholder="タスクを検索..."
                               x-model="filters.search">
                    </div>
                    
                    <!-- ステータス -->
                    <div class="col-md-2">
                        <label class="form-label">ステータス</label>
                        <select class="form-select" name="status" x-model="filters.status">
                            <option value="">すべて</option>
                            @foreach($statusOptions as $value => $label)
                                <option value="{{ $value }}" 
                                        {{ request('status') == $value ? 'selected' : '' }}>
                                    {{ $label }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    
                    <!-- 優先度 -->
                    <div class="col-md-2">
                        <label class="form-label">優先度</label>
                        <select class="form-select" name="priority" x-model="filters.priority">
                            <option value="">すべて</option>
                            @foreach($priorityOptions as $value => $label)
                                <option value="{{ $value }}" 
                                        {{ request('priority') == $value ? 'selected' : '' }}>
                                    {{ $label }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    
                    <!-- カテゴリ -->
                    <div class="col-md-2">
                        <label class="form-label">カテゴリ</label>
                        <select class="form-select" name="category_id" x-model="filters.category">
                            <option value="">すべて</option>
                            @foreach($categories as $category)
                                <option value="{{ $category->id }}" 
                                        {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                    {{ $category->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    
                    <!-- 期限日 -->
                    <div class="col-md-2">
                        <label class="form-label">期限日（〜まで）</label>
                        <input type="date" class="form-control" name="due_date_to" 
                               value="{{ request('due_date_to') }}"
                               x-model="filters.dueDate">
                    </div>
                    
                    <!-- アクションボタン -->
                    <div class="col-md-1">
                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary btn-sm" title="検索">
                                <i class="fas fa-search"></i>
                            </button>
                            <a href="{{ route('tasks.index') }}" class="btn btn-outline-secondary btn-sm" title="リセット">
                                <i class="fas fa-times"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- ソート・表示オプション -->
    <div class="d-flex justify-content-between align-items-center mb-3">
        <div class="d-flex align-items-center gap-3">
            <!-- ソート -->
            <div class="dropdown">
                <button class="btn btn-outline-secondary dropdown-toggle btn-sm" type="button" 
                        data-bs-toggle="dropdown">
                    <i class="fas fa-sort me-1"></i>並び順
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="{{ request()->fullUrlWithQuery(['sort_by' => 'created_at', 'sort_dir' => 'desc']) }}">
                        <i class="fas fa-clock me-2"></i>作成日（新しい順）
                    </a></li>
                    <li><a class="dropdown-item" href="{{ request()->fullUrlWithQuery(['sort_by' => 'created_at', 'sort_dir' => 'asc']) }}">
                        <i class="fas fa-clock me-2"></i>作成日（古い順）
                    </a></li>
                    <li><a class="dropdown-item" href="{{ request()->fullUrlWithQuery(['sort_by' => 'due_date', 'sort_dir' => 'asc']) }}">
                        <i class="fas fa-calendar me-2"></i>期限日（近い順）
                    </a></li>
                    <li><a class="dropdown-item" href="{{ request()->fullUrlWithQuery(['sort_by' => 'priority', 'sort_dir' => 'desc']) }}">
                        <i class="fas fa-exclamation-triangle me-2"></i>優先度（高い順）
                    </a></li>
                </ul>
            </div>
            
            <!-- 表示形式切り替え -->
            <div class="btn-group" role="group">
                <input type="radio" class="btn-check" name="view_mode" id="card_view" 
                       x-model="viewMode" value="card" checked>
                <label class="btn btn-outline-secondary btn-sm" for="card_view">
                    <i class="fas fa-th-large"></i>
                </label>
                
                <input type="radio" class="btn-check" name="view_mode" id="table_view" 
                       x-model="viewMode" value="table">
                <label class="btn btn-outline-secondary btn-sm" for="table_view">
                    <i class="fas fa-list"></i>
                </label>
            </div>
        </div>
        
        <!-- 一括操作 -->
        <div x-show="selectedTasks.length > 0" x-transition class="d-flex align-items-center gap-2">
            <span class="text-muted" x-text="selectedTasks.length + '件選択中'"></span>
            <div class="btn-group">
                <button type="button" class="btn btn-outline-primary btn-sm" 
                        @click="bulkComplete()" 
                        :disabled="processing">
                    <i class="fas fa-check me-1"></i>完了にする
                </button>
                <button type="button" class="btn btn-outline-danger btn-sm" 
                        @click="bulkDelete()" 
                        :disabled="processing">
                    <i class="fas fa-trash me-1"></i>削除
                </button>
            </div>
        </div>
    </div>

    <!-- タスク一覧（カード表示） -->
    <div x-show="viewMode === 'card'" x-transition>
        @if($tasks->count() > 0)
            <div class="row g-3">
                @foreach($tasks as $task)
                    <div class="col-md-6 col-lg-4">
                        <div class="card h-100 task-card" 
                             :class="selectedTasks.includes({{ $task->id }}) ? 'border-primary' : ''">
                            <div class="card-header d-flex justify-content-between align-items-start py-2">
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" 
                                           value="{{ $task->id }}" 
                                           x-model="selectedTasks">
                                </div>
                                <div class="d-flex gap-1">
                                    <span class="badge task-priority-{{ $task->priority }}">
                                        {{ $task->priority_display }}
                                    </span>
                                    <span class="badge bg-secondary task-status-{{ $task->status }}">
                                        {{ $task->status_display }}
                                    </span>
                                </div>
                            </div>
                            
                            <div class="card-body">
                                <h6 class="card-title">
                                    <a href="{{ route('tasks.show', $task) }}" class="text-decoration-none">
                                        {{ Str::limit($task->title, 40) }}
                                    </a>
                                </h6>
                                
                                @if($task->description)
                                    <p class="card-text text-muted small">
                                        {{ Str::limit($task->description, 80) }}
                                    </p>
                                @endif
                                
                                <!-- 進捗バー -->
                                @if($task->progress > 0)
                                    <div class="mb-2">
                                        <div class="d-flex justify-content-between mb-1">
                                            <small class="text-muted">進捗</small>
                                            <small class="text-muted">{{ $task->progress }}%</small>
                                        </div>
                                        <div class="progress" style="height: 4px;">
                                            <div class="progress-bar" style="width: {{ $task->progress }}%"></div>
                                        </div>
                                    </div>
                                @endif
                                
                                <!-- メタ情報 -->
                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted">
                                        <i class="fas fa-user me-1"></i>{{ $task->user->name }}
                                    </small>
                                    @if($task->due_date)
                                        <small class="text-muted {{ $task->is_overdue ? 'text-danger' : '' }}">
                                            <i class="fas fa-calendar me-1"></i>
                                            {{ $task->due_date->format('m/d') }}
                                            @if($task->is_overdue)
                                                <i class="fas fa-exclamation-triangle text-danger ms-1"></i>
                                            @endif
                                        </small>
                                    @endif
                                </div>
                            </div>
                            
                            <!-- アクションボタン -->
                            <div class="card-footer bg-transparent py-2">
                                <div class="d-flex justify-content-end gap-2">
                                    <a href="{{ route('tasks.show', $task) }}" 
                                       class="btn btn-outline-primary btn-sm">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <a href="{{ route('tasks.edit', $task) }}" 
                                       class="btn btn-outline-secondary btn-sm">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <button type="button" 
                                            class="btn btn-outline-danger btn-sm" 
                                            @click="deleteTask({{ $task->id }})">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                @endforeach
            </div>
        @else
            <div class="text-center py-5">
                <i class="fas fa-tasks text-muted" style="font-size: 4rem;"></i>
                <h4 class="text-muted mt-3">タスクが見つかりません</h4>
                <p class="text-muted">新しいタスクを作成してみましょう。</p>
                <a href="{{ route('tasks.create') }}" class="btn btn-primary">
                    <i class="fas fa-plus me-1"></i>新規タスク作成
                </a>
            </div>
        @endif
    </div>

    <!-- タスク一覧（テーブル表示） -->
    <div x-show="viewMode === 'table'" x-transition>
        @if($tasks->count() > 0)
            <div class="card">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th width="40">
                                    <input type="checkbox" class="form-check-input" 
                                           @change="selectAll($event.target.checked)">
                                </th>
                                <th>タイトル</th>
                                <th width="100">優先度</th>
                                <th width="100">ステータス</th>
                                <th width="120">担当者</th>
                                <th width="120">期限日</th>
                                <th width="80">進捗</th>
                                <th width="150">アクション</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($tasks as $task)
                                <tr :class="selectedTasks.includes({{ $task->id }}) ? 'table-primary' : ''">
                                    <td>
                                        <input type="checkbox" class="form-check-input" 
                                               value="{{ $task->id }}" 
                                               x-model="selectedTasks">
                                    </td>
                                    <td>
                                        <a href="{{ route('tasks.show', $task) }}" 
                                           class="text-decoration-none fw-semibold">
                                            {{ $task->title }}
                                        </a>
                                        @if($task->description)
                                            <br><small class="text-muted">{{ Str::limit($task->description, 60) }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        <span class="badge task-priority-{{ $task->priority }}">
                                            {{ $task->priority_display }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge bg-secondary task-status-{{ $task->status }}">
                                            {{ $task->status_display }}
                                        </span>
                                    </td>
                                    <td>
                                        <i class="fas fa-user me-1"></i>{{ $task->user->name }}
                                    </td>
                                    <td>
                                        @if($task->due_date)
                                            <span class="{{ $task->is_overdue ? 'text-danger' : '' }}">
                                                {{ $task->due_date->format('Y/m/d') }}
                                                @if($task->is_overdue)
                                                    <i class="fas fa-exclamation-triangle ms-1"></i>
                                                @endif
                                            </span>
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($task->progress > 0)
                                            <div class="d-flex align-items-center">
                                                <div class="progress me-2" style="width: 60px; height: 8px;">
                                                    <div class="progress-bar" style="width: {{ $task->progress }}%"></div>
                                                </div>
                                                <small>{{ $task->progress }}%</small>
                                            </div>
                                        @else
                                            <span class="text-muted">0%</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="btn-group">
                                            <a href="{{ route('tasks.show', $task) }}" 
                                               class="btn btn-outline-primary btn-sm">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('tasks.edit', $task) }}" 
                                               class="btn btn-outline-secondary btn-sm">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <button type="button" 
                                                    class="btn btn-outline-danger btn-sm" 
                                                    @click="deleteTask({{ $task->id }})">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        @else
            <div class="text-center py-5">
                <i class="fas fa-tasks text-muted" style="font-size: 4rem;"></i>
                <h4 class="text-muted mt-3">タスクが見つかりません</h4>
                <p class="text-muted">新しいタスクを作成してみましょう。</p>
                <a href="{{ route('tasks.create') }}" class="btn btn-primary">
                    <i class="fas fa-plus me-1"></i>新規タスク作成
                </a>
            </div>
        @endif
    </div>

    <!-- ページネーション -->
    @if($tasks->hasPages())
        <div class="d-flex justify-content-center mt-4">
            {{ $tasks->withQueryString()->links() }}
        </div>
    @endif
</div>

<!-- 削除確認モーダル -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">タスクの削除確認</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>選択されたタスクを削除してもよろしいですか？</p>
                <p class="text-danger small">
                    <i class="fas fa-exclamation-triangle me-1"></i>
                    この操作は元に戻すことができません。
                </p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                <button type="button" class="btn btn-danger" id="confirmDelete">
                    <i class="fas fa-trash me-1"></i>削除する
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function taskIndex() {
    return {
        viewMode: 'card',
        selectedTasks: [],
        processing: false,
        filters: {
            search: '{{ request("search") }}',
            status: '{{ request("status") }}',
            priority: '{{ request("priority") }}',
            category: '{{ request("category_id") }}',
            dueDate: '{{ request("due_date_to") }}'
        },

        selectAll(checked) {
            if (checked) {
                this.selectedTasks = Array.from(document.querySelectorAll('input[type="checkbox"][value]'))
                    .map(checkbox => parseInt(checkbox.value));
            } else {
                this.selectedTasks = [];
            }
        },

        async bulkComplete() {
            if (!this.selectedTasks.length) return;
            
            this.processing = true;
            
            try {
                const response = await fetch('{{ route("tasks.bulk-action") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        task_ids: this.selectedTasks,
                        action: 'complete'
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    window.location.reload();
                } else {
                    alert('エラーが発生しました: ' + result.message);
                }
            } catch (error) {
                alert('通信エラーが発生しました');
            } finally {
                this.processing = false;
            }
        },

        async bulkDelete() {
            if (!this.selectedTasks.length) return;
            
            if (!confirm(`${this.selectedTasks.length}件のタスクを削除してもよろしいですか？`)) {
                return;
            }
            
            this.processing = true;
            
            try {
                const response = await fetch('{{ route("tasks.bulk-action") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        task_ids: this.selectedTasks,
                        action: 'delete'
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    window.location.reload();
                } else {
                    alert('エラーが発生しました: ' + result.message);
                }
            } catch (error) {
                alert('通信エラーが発生しました');
            } finally {
                this.processing = false;
            }
        },

        deleteTask(taskId) {
            if (confirm('このタスクを削除してもよろしいですか？')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = `/tasks/${taskId}`;
                form.innerHTML = `
                    @csrf
                    @method('DELETE')
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
    };
}
</script>
@endpush