<?php
/**
 * 第6章 サンプルコントローラー - TaskController（ユーザー固有データ対応版）
 * 
 * 認証・認可・セキュリティ機能を統合したタスク管理コントローラー
 * - ユーザー固有データの自動制御
 * - Policy による認可制御
 * - セキュリティ対策の実装
 */

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\Category;
use App\Models\UserActivity;
use App\Http\Requests\StoreTaskRequest;
use App\Http\Requests\UpdateTaskRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\JsonResponse;

class TaskController extends Controller
{
    /**
     * コンストラクタ - 認証必須の設定
     */
    public function __construct()
    {
        // すべてのメソッドで認証を必須とする
        $this->middleware('auth');
        
        // レート制限の適用
        $this->middleware('throttle:60,1')->only(['index', 'show']);
        $this->middleware('throttle:30,1')->only(['store', 'update', 'destroy']);
    }

    /**
     * 認証済みユーザーのタスク一覧表示
     * 
     * ✅ セキュリティポイント：
     * - 自動的に現在のユーザーのタスクのみ表示
     * - 他ユーザーのデータは一切取得されない
     * - XSS対策済みの検索機能
     */
    public function index(Request $request)
    {
        // ユーザー活動の記録
        $this->logUserActivity('task.index', 'タスク一覧表示');

        // 現在認証済みユーザーのタスクのみクエリ
        $query = Auth::user()->tasks()->with(['category', 'user:id,name']);

        // 検索フィルタ（入力値は自動エスケープされる）
        if ($request->filled('search')) {
            $searchTerm = strip_tags($request->search); // HTMLタグ除去
            $query->where(function($q) use ($searchTerm) {
                $q->where('title', 'LIKE', '%' . $searchTerm . '%')
                  ->orWhere('description', 'LIKE', '%' . $searchTerm . '%');
            });
        }

        // ステータスフィルタ
        if ($request->filled('completed')) {
            $query->where('completed', $request->boolean('completed'));
        }

        // 優先度フィルタ
        if ($request->filled('priority')) {
            $priority = in_array($request->priority, ['low', 'medium', 'high']) 
                       ? $request->priority 
                       : null;
            if ($priority) {
                $query->where('priority', $priority);
            }
        }

        // カテゴリフィルタ（ユーザーが作成したカテゴリのみ）
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id)
                  ->whereHas('category', function($q) {
                      $q->where('user_id', Auth::id());
                  });
        }

        // 期限フィルタ
        if ($request->filled('due_date_from')) {
            $query->where('due_date', '>=', $request->due_date_from);
        }
        if ($request->filled('due_date_to')) {
            $query->where('due_date', '<=', $request->due_date_to);
        }

        // ソート設定
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');
        
        // ソートフィールドのホワイトリスト（SQLインジェクション対策）
        $allowedSortFields = ['title', 'created_at', 'updated_at', 'due_date', 'priority', 'completed'];
        if (in_array($sortField, $allowedSortFields)) {
            $query->orderBy($sortField, $sortDirection === 'asc' ? 'asc' : 'desc');
        } else {
            $query->latest('created_at');
        }

        $tasks = $query->paginate(15)->withQueryString();
        
        // ユーザーが作成したカテゴリのみ取得
        $categories = Auth::user()->categories()->orderBy('name')->get();
        
        // 統計情報の取得
        $stats = $this->getTaskStats();

        return view('tasks.index', compact('tasks', 'categories', 'stats'));
    }

    /**
     * 新規タスク作成フォーム表示
     */
    public function create()
    {
        // 作成権限のチェック
        $this->authorize('create', Task::class);

        $this->logUserActivity('task.create_form', 'タスク作成フォーム表示');

        // ユーザー固有のカテゴリのみ提供
        $categories = Auth::user()->categories()->orderBy('name')->get();
        
        return view('tasks.create', compact('categories'));
    }

    /**
     * タスク作成処理
     * 
     * ✅ セキュリティポイント：
     * - 自動的に現在ユーザーIDを設定
     * - 他ユーザーになりすましての作成を防止
     * - フォームリクエストによる厳密なバリデーション
     */
    public function store(StoreTaskRequest $request)
    {
        try {
            // 認証済みユーザーのタスクとして作成
            $task = Auth::user()->tasks()->create([
                'title' => $request->title,
                'description' => $request->description,
                'category_id' => $request->category_id,
                'due_date' => $request->due_date,
                'priority' => $request->priority,
                'is_private' => $request->boolean('is_private', true), // デフォルトはプライベート
                'tags' => $request->tags ? json_encode($request->tags) : null,
            ]);

            // ユーザー活動の記録
            $this->logUserActivity('task.created', "タスク作成: {$task->title}", [
                'task_id' => $task->id,
                'priority' => $task->priority,
                'due_date' => $task->due_date,
            ]);

            // ログ記録
            Log::info('タスクが作成されました', [
                'user_id' => Auth::id(),
                'task_id' => $task->id,
                'task_title' => $task->title,
            ]);

            return redirect()->route('tasks.show', $task)
                            ->with('success', 'タスクが正常に作成されました。');

        } catch (\Exception $e) {
            // エラーログ記録
            Log::error('タスク作成エラー', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
                'request_data' => $request->except(['_token']),
            ]);

            return back()
                   ->withInput()
                   ->withErrors(['error' => 'タスクの作成中にエラーが発生しました。']);
        }
    }

    /**
     * タスク詳細表示
     * 
     * ✅ セキュリティポイント：
     * - Route Model Bindingで自動認可
     * - 他ユーザーのタスクはアクセス不可
     */
    public function show(Task $task)
    {
        // Policy による認可チェック
        $this->authorize('view', $task);

        $this->logUserActivity('task.viewed', "タスク詳細表示: {$task->title}", [
            'task_id' => $task->id,
        ]);
        
        // 関連データの eager loading
        $task->load(['category', 'user:id,name']);
        
        return view('tasks.show', compact('task'));
    }

    /**
     * タスク編集フォーム表示
     */
    public function edit(Task $task)
    {
        $this->authorize('update', $task);

        $this->logUserActivity('task.edit_form', "タスク編集フォーム表示: {$task->title}", [
            'task_id' => $task->id,
        ]);
        
        // ユーザー固有のカテゴリのみ提供
        $categories = Auth::user()->categories()->orderBy('name')->get();
        
        return view('tasks.edit', compact('task', 'categories'));
    }

    /**
     * タスク更新処理
     * 
     * ✅ セキュリティポイント：
     * - Policy による所有権チェック
     * - 他ユーザーのタスクは更新不可
     */
    public function update(UpdateTaskRequest $request, Task $task)
    {
        $this->authorize('update', $task);

        try {
            // 更新前のデータを記録（監査用）
            $originalData = $task->toArray();

            $task->update([
                'title' => $request->title,
                'description' => $request->description,
                'category_id' => $request->category_id,
                'due_date' => $request->due_date,
                'priority' => $request->priority,
                'completed' => $request->boolean('completed'),
                'is_private' => $request->boolean('is_private'),
                'tags' => $request->tags ? json_encode($request->tags) : null,
            ]);

            // 変更内容の記録
            $changes = $task->getChanges();
            $this->logUserActivity('task.updated', "タスク更新: {$task->title}", [
                'task_id' => $task->id,
                'changes' => $changes,
                'original' => $originalData,
            ]);

            Log::info('タスクが更新されました', [
                'user_id' => Auth::id(),
                'task_id' => $task->id,
                'changes' => $changes,
            ]);

            return redirect()->route('tasks.show', $task)
                            ->with('success', 'タスクが正常に更新されました。');

        } catch (\Exception $e) {
            Log::error('タスク更新エラー', [
                'user_id' => Auth::id(),
                'task_id' => $task->id,
                'error' => $e->getMessage(),
            ]);

            return back()
                   ->withInput()
                   ->withErrors(['error' => 'タスクの更新中にエラーが発生しました。']);
        }
    }

    /**
     * タスク削除処理
     * 
     * ✅ セキュリティポイント：
     * - Policy による所有権チェック
     * - ソフトデリートによる安全な削除
     */
    public function destroy(Task $task)
    {
        $this->authorize('delete', $task);

        try {
            $taskTitle = $task->title;
            $taskId = $task->id;

            // ソフトデリート実行
            $task->delete();

            $this->logUserActivity('task.deleted', "タスク削除: {$taskTitle}", [
                'task_id' => $taskId,
                'deleted_at' => now(),
            ]);

            Log::info('タスクが削除されました', [
                'user_id' => Auth::id(),
                'task_id' => $taskId,
                'task_title' => $taskTitle,
            ]);

            return redirect()->route('tasks.index')
                            ->with('success', 'タスクが正常に削除されました。');

        } catch (\Exception $e) {
            Log::error('タスク削除エラー', [
                'user_id' => Auth::id(),
                'task_id' => $task->id,
                'error' => $e->getMessage(),
            ]);

            return back()
                   ->withErrors(['error' => 'タスクの削除中にエラーが発生しました。']);
        }
    }

    /**
     * タスク完了状態の切り替え（Ajax対応）
     * 
     * ✅ セキュリティポイント：
     * - CSRF保護
     * - JSON レスポンス
     */
    public function toggleStatus(Request $request, Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        try {
            $completed = $request->boolean('completed');
            $task->update(['completed' => $completed]);

            $this->logUserActivity('task.status_toggled', "タスクステータス変更: {$task->title}", [
                'task_id' => $task->id,
                'completed' => $completed,
            ]);

            return response()->json([
                'success' => true,
                'message' => $completed ? 'タスクを完了にしました。' : 'タスクを未完了にしました。',
                'completed' => $task->completed,
                'updated_at' => $task->updated_at->format('Y-m-d H:i:s'),
            ]);

        } catch (\Exception $e) {
            Log::error('タスクステータス更新エラー', [
                'user_id' => Auth::id(),
                'task_id' => $task->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'ステータスの更新に失敗しました。',
            ], 500);
        }
    }

    /**
     * ユーザーダッシュボード表示
     */
    public function dashboard()
    {
        $user = Auth::user();
        
        // ユーザー統計情報
        $stats = $this->getTaskStats();
        
        // 最近の活動（最新10件）
        $recentTasks = $user->tasks()
                           ->latest('updated_at')
                           ->limit(10)
                           ->get();
        
        // 期限が近いタスク（今後7日間）
        $upcomingTasks = $user->tasks()
                             ->where('due_date', '>=', now())
                             ->where('due_date', '<=', now()->addDays(7))
                             ->where('completed', false)
                             ->orderBy('due_date')
                             ->limit(5)
                             ->with('category')
                             ->get();

        // 優先度別タスク数
        $priorityStats = $user->tasks()
                             ->selectRaw('priority, COUNT(*) as count')
                             ->where('completed', false)
                             ->groupBy('priority')
                             ->pluck('count', 'priority')
                             ->toArray();

        $this->logUserActivity('dashboard.viewed', 'ダッシュボード表示');
        
        return view('dashboard', compact('stats', 'recentTasks', 'upcomingTasks', 'priorityStats'));
    }

    /**
     * タスクの一括操作（選択したタスクの一括処理）
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'task_ids' => 'required|array',
            'task_ids.*' => 'integer|exists:tasks,id',
            'action' => 'required|in:complete,incomplete,delete',
        ]);

        try {
            $taskIds = $request->task_ids;
            $action = $request->action;

            // ユーザーのタスクのみ取得（セキュリティチェック）
            $tasks = Auth::user()->tasks()->whereIn('id', $taskIds)->get();

            if ($tasks->count() !== count($taskIds)) {
                return back()->withErrors(['error' => '不正なタスクIDが含まれています。']);
            }

            $affectedCount = 0;

            foreach ($tasks as $task) {
                switch ($action) {
                    case 'complete':
                        if ($this->authorize('update', $task)) {
                            $task->update(['completed' => true]);
                            $affectedCount++;
                        }
                        break;
                    case 'incomplete':
                        if ($this->authorize('update', $task)) {
                            $task->update(['completed' => false]);
                            $affectedCount++;
                        }
                        break;
                    case 'delete':
                        if ($this->authorize('delete', $task)) {
                            $task->delete();
                            $affectedCount++;
                        }
                        break;
                }
            }

            $this->logUserActivity('task.bulk_action', "一括操作実行: {$action}", [
                'action' => $action,
                'task_ids' => $taskIds,
                'affected_count' => $affectedCount,
            ]);

            $actionName = [
                'complete' => '完了',
                'incomplete' => '未完了',
                'delete' => '削除',
            ][$action];

            return back()->with('success', "{$affectedCount}件のタスクを{$actionName}にしました。");

        } catch (\Exception $e) {
            Log::error('一括操作エラー', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
                'request_data' => $request->all(),
            ]);

            return back()->withErrors(['error' => '一括操作中にエラーが発生しました。']);
        }
    }

    /**
     * ユーザー統計情報を取得
     * 
     * @return array
     */
    private function getTaskStats(): array
    {
        $user = Auth::user();
        $totalTasks = $user->tasks()->count();
        $completedTasks = $user->tasks()->where('completed', true)->count();
        $pendingTasks = $totalTasks - $completedTasks;

        return [
            'total_tasks' => $totalTasks,
            'completed_tasks' => $completedTasks,
            'pending_tasks' => $pendingTasks,
            'completion_rate' => $totalTasks > 0 
                ? round(($completedTasks / $totalTasks) * 100, 1)
                : 0,
            'overdue_tasks' => $user->tasks()
                                  ->where('completed', false)
                                  ->where('due_date', '<', now())
                                  ->count(),
            'today_tasks' => $user->tasks()
                                 ->whereDate('due_date', now())
                                 ->count(),
        ];
    }

    /**
     * ユーザー活動をログに記録
     * 
     * @param string $action
     * @param string $description
     * @param array $metadata
     */
    private function logUserActivity(string $action, string $description, array $metadata = []): void
    {
        try {
            UserActivity::create([
                'user_id' => Auth::id(),
                'action' => $action,
                'description' => $description,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'metadata' => json_encode($metadata),
                'created_at' => now(),
            ]);
        } catch (\Exception $e) {
            // ユーザー活動のログ記録に失敗してもアプリケーションは継続
            Log::error('ユーザー活動ログ記録エラー', [
                'user_id' => Auth::id(),
                'action' => $action,
                'error' => $e->getMessage(),
            ]);
        }
    }
}