<?php
/**
 * 第6章 サンプルリクエスト - StoreTaskRequest
 * 
 * タスク作成用のフォームリクエスト
 * - 厳密なバリデーション実装
 * - XSS対策を含む入力値の前処理
 * - セキュリティを考慮したバリデーションルール
 */

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class StoreTaskRequest extends FormRequest
{
    /**
     * 認可チェック
     * 
     * ✅ セキュリティポイント：
     * - 認証済みユーザーのみタスク作成可能
     * - 追加の権限チェックも可能
     */
    public function authorize(): bool
    {
        // 認証済みユーザーのみタスク作成を許可
        return Auth::check();
    }

    /**
     * バリデーションルール定義
     * 
     * ✅ セキュリティポイント：
     * - 各フィールドに対する厳密な検証
     * - SQLインジェクション対策
     * - データ型・長さ・形式の厳密なチェック
     */
    public function rules(): array
    {
        return [
            // タスクタイトル
            'title' => [
                'required',
                'string',
                'max:255',
                'min:1',
                // 同一ユーザー内でのタイトル重複チェック
                Rule::unique('tasks')->where(function ($query) {
                    return $query->where('user_id', Auth::id());
                }),
            ],

            // タスク説明
            'description' => [
                'nullable',
                'string',
                'max:2000', // 適切な長さ制限
            ],

            // カテゴリID（ユーザーが所有するカテゴリのみ）
            'category_id' => [
                'required',
                'integer',
                Rule::exists('categories', 'id')->where(function ($query) {
                    return $query->where('user_id', Auth::id());
                }),
            ],

            // 期限日
            'due_date' => [
                'nullable',
                'date',
                'after_or_equal:today', // 過去の日付は不可
                'before:' . now()->addYears(5)->format('Y-m-d'), // 5年以内
            ],

            // 優先度
            'priority' => [
                'required',
                'string',
                Rule::in(['low', 'medium', 'high']), // ホワイトリスト方式
            ],

            // プライベート設定
            'is_private' => [
                'boolean',
            ],

            // タグ（配列形式）
            'tags' => [
                'nullable',
                'array',
                'max:10', // 最大10個のタグ
            ],
            'tags.*' => [
                'string',
                'max:50', // 各タグの最大長
                'regex:/^[a-zA-Z0-9\s\-_]+$/u', // 許可文字の制限
            ],

            // 添付ファイル（将来の拡張用）
            'attachments' => [
                'nullable',
                'array',
                'max:5', // 最大5ファイル
            ],
            'attachments.*' => [
                'file',
                'max:10240', // 10MB制限
                'mimes:pdf,doc,docx,txt,jpg,png,gif', // ファイル形式制限
            ],
        ];
    }

    /**
     * 入力データの前処理
     * 
     * ✅ セキュリティポイント：
     * - HTMLタグの除去（XSS対策）
     * - 不正な文字の除去
     * - データの正規化
     */
    protected function prepareForValidation(): void
    {
        $this->merge([
            // タイトルのHTMLタグ除去とトリム
            'title' => $this->title ? trim(strip_tags($this->title)) : null,
            
            // 説明文のHTMLタグ除去（改行は保持）
            'description' => $this->description ? $this->sanitizeDescription($this->description) : null,
            
            // 優先度の正規化
            'priority' => $this->priority ? strtolower(trim($this->priority)) : 'medium',
            
            // プライベート設定のデフォルト値
            'is_private' => $this->has('is_private') ? $this->boolean('is_private') : true,
            
            // タグの前処理
            'tags' => $this->tags ? $this->sanitizeTags($this->tags) : null,
        ]);
    }

    /**
     * 説明文の安全な処理
     * 
     * @param string $description
     * @return string
     */
    private function sanitizeDescription(string $description): string
    {
        // HTMLタグを除去しつつ改行を保持
        $description = strip_tags($description);
        
        // 連続する空白を単一スペースに変換
        $description = preg_replace('/\s+/', ' ', $description);
        
        // 連続する改行を最大2つに制限
        $description = preg_replace('/\n{3,}/', "\n\n", $description);
        
        return trim($description);
    }

    /**
     * タグの安全な処理
     * 
     * @param array $tags
     * @return array
     */
    private function sanitizeTags(array $tags): array
    {
        return array_filter(array_map(function ($tag) {
            if (!is_string($tag)) {
                return null;
            }
            
            // HTMLタグ除去とトリム
            $tag = trim(strip_tags($tag));
            
            // 空文字や短すぎるタグは除外
            if (strlen($tag) < 2) {
                return null;
            }
            
            // 小文字に統一
            $tag = strtolower($tag);
            
            // 不正な文字を除去
            $tag = preg_replace('/[^a-zA-Z0-9\s\-_]/', '', $tag);
            
            return $tag;
        }, $tags));
    }

    /**
     * カスタム属性名
     * 
     * エラーメッセージで使用される属性名を日本語に変換
     */
    public function attributes(): array
    {
        return [
            'title' => 'タスクタイトル',
            'description' => '説明',
            'category_id' => 'カテゴリ',
            'due_date' => '期限',
            'priority' => '優先度',
            'is_private' => 'プライベート設定',
            'tags' => 'タグ',
            'tags.*' => 'タグ',
            'attachments' => '添付ファイル',
            'attachments.*' => '添付ファイル',
        ];
    }

    /**
     * カスタムエラーメッセージ
     * 
     * より分かりやすい日本語エラーメッセージを提供
     */
    public function messages(): array
    {
        return [
            // タイトル関連
            'title.required' => 'タスクタイトルは必須です。',
            'title.string' => 'タスクタイトルは文字列で入力してください。',
            'title.max' => 'タスクタイトルは255文字以下で入力してください。',
            'title.min' => 'タスクタイトルは1文字以上で入力してください。',
            'title.unique' => 'このタスクタイトルは既に存在します。',

            // 説明関連
            'description.string' => '説明は文字列で入力してください。',
            'description.max' => '説明は2000文字以下で入力してください。',

            // カテゴリ関連
            'category_id.required' => 'カテゴリの選択は必須です。',
            'category_id.integer' => 'カテゴリIDは整数で指定してください。',
            'category_id.exists' => '選択されたカテゴリは存在しないか、アクセス権限がありません。',

            // 期限関連
            'due_date.date' => '期限は有効な日付を入力してください。',
            'due_date.after_or_equal' => '期限は今日以降の日付を指定してください。',
            'due_date.before' => '期限は5年以内の日付を指定してください。',

            // 優先度関連
            'priority.required' => '優先度の選択は必須です。',
            'priority.in' => '優先度は「低」「中」「高」のいずれかを選択してください。',

            // プライベート設定関連
            'is_private.boolean' => 'プライベート設定は真偽値で指定してください。',

            // タグ関連
            'tags.array' => 'タグは配列形式で指定してください。',
            'tags.max' => 'タグは最大10個まで設定できます。',
            'tags.*.string' => '各タグは文字列で入力してください。',
            'tags.*.max' => '各タグは50文字以下で入力してください。',
            'tags.*.regex' => 'タグには英数字、スペース、ハイフン、アンダースコアのみ使用できます。',

            // 添付ファイル関連
            'attachments.array' => '添付ファイルは配列形式で指定してください。',
            'attachments.max' => '添付ファイルは最大5個まで追加できます。',
            'attachments.*.file' => '添付ファイルが有効なファイルではありません。',
            'attachments.*.max' => '添付ファイルのサイズは10MB以下にしてください。',
            'attachments.*.mimes' => '添付ファイルの形式が対応していません。対応形式：PDF, DOC, DOCX, TXT, JPG, PNG, GIF',
        ];
    }

    /**
     * バリデーション後の追加処理
     * 
     * バリデーション成功後に実行される処理
     */
    protected function passedValidation(): void
    {
        // ログ記録（セキュリティ監査用）
        \Log::info('タスク作成バリデーション通過', [
            'user_id' => Auth::id(),
            'title' => $this->title,
            'category_id' => $this->category_id,
            'priority' => $this->priority,
            'ip_address' => $this->ip(),
            'user_agent' => $this->userAgent(),
        ]);
    }

    /**
     * バリデーション失敗時の追加処理
     * 
     * @param \Illuminate\Contracts\Validation\Validator $validator
     */
    protected function failedValidation(\Illuminate\Contracts\Validation\Validator $validator): void
    {
        // セキュリティログ記録（不正入力の試行記録）
        \Log::warning('タスク作成バリデーション失敗', [
            'user_id' => Auth::id(),
            'errors' => $validator->errors()->toArray(),
            'input_data' => $this->except(['_token', 'password', 'attachments']),
            'ip_address' => $this->ip(),
            'user_agent' => $this->userAgent(),
        ]);

        parent::failedValidation($validator);
    }

    /**
     * バリデーション済みデータに追加データを含める
     * 
     * @return array
     */
    public function validatedWithExtras(): array
    {
        $validated = $this->validated();
        
        // システムが自動設定する値を追加
        $validated['user_id'] = Auth::id();
        $validated['created_at'] = now();
        $validated['updated_at'] = now();
        
        return $validated;
    }
}