<?php
/**
 * 第7章 サンプルテスト - TaskManagementTest
 * 
 * タスク管理機能の包括的なFeatureテスト
 * - CRUD操作の完全テスト
 * - 認証・認可制御のテスト
 * - バリデーション・エラーハンドリングのテスト
 * - 検索・フィルタリング・ページネーションのテスト
 */

namespace Tests\Feature;

use App\Models\Task;
use App\Models\User;
use App\Models\Category;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class TaskManagementTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected User $user;
    protected User $otherUser;
    protected Category $category;

    /**
     * 各テスト前のセットアップ
     */
    protected function setUp(): void
    {
        parent::setUp();
        
        // テスト用ユーザーとカテゴリを作成
        $this->user = User::factory()->create([
            'name' => 'テストユーザー',
            'email' => 'test@example.com',
        ]);
        
        $this->otherUser = User::factory()->create([
            'name' => '他のユーザー',
            'email' => 'other@example.com',
        ]);
        
        $this->category = Category::factory()->for($this->user)->create([
            'name' => 'テストカテゴリ',
        ]);
    }

    /**
     * 認証済みユーザーがタスク一覧を表示できることをテスト
     */
    public function test_authenticated_user_can_view_task_list(): void
    {
        // テストデータの準備
        $tasks = Task::factory(3)->for($this->user)->create([
            'title' => fn() => $this->faker->sentence(3),
        ]);

        // 他のユーザーのタスクも作成（表示されないことを確認するため）
        Task::factory(2)->for($this->otherUser)->create();

        // リクエスト実行
        $response = $this->actingAs($this->user)->get('/tasks');

        // レスポンスの検証
        $response->assertStatus(200);
        $response->assertViewIs('tasks.index');
        
        // 自分のタスクのみが表示されていることを確認
        foreach ($tasks as $task) {
            $response->assertSee($task->title);
        }
        
        // タスク数の確認
        $response->assertViewHas('tasks', function ($viewTasks) {
            return $viewTasks->count() === 3;
        });
    }

    /**
     * 未認証ユーザーがタスク一覧にアクセスできないことをテスト
     */
    public function test_guest_user_cannot_view_task_list(): void
    {
        $response = $this->get('/tasks');

        $response->assertRedirect('/login');
        $response->assertStatus(302);
    }

    /**
     * タスクの作成が正常に動作することをテスト
     */
    public function test_user_can_create_task_with_valid_data(): void
    {
        $taskData = [
            'title' => 'Laravel テスト実装',
            'description' => 'Feature Test の実装を完了する',
            'category_id' => $this->category->id,
            'due_date' => now()->addWeeks(2)->format('Y-m-d'),
            'priority' => 'high',
            'is_private' => true,
            'tags' => ['laravel', 'testing', 'php'],
        ];

        $response = $this->actingAs($this->user)
                        ->post('/tasks', $taskData);

        // リダイレクトの確認
        $response->assertStatus(302);
        
        // データベースに正しく保存されていることを確認
        $this->assertDatabaseHas('tasks', [
            'title' => 'Laravel テスト実装',
            'description' => 'Feature Test の実装を完了する',
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
            'priority' => 'high',
            'is_private' => true,
            'completed' => false,
        ]);

        // 作成されたタスクの取得と詳細確認
        $createdTask = Task::where('user_id', $this->user->id)
                          ->where('title', 'Laravel テスト実装')
                          ->first();
        
        $this->assertNotNull($createdTask);
        $this->assertEquals(['laravel', 'testing', 'php'], json_decode($createdTask->tags));
        
        // 成功メッセージの確認
        $response->assertSessionHas('success');
    }

    /**
     * バリデーションエラーが正しく表示されることをテスト
     */
    public function test_task_creation_validates_required_fields(): void
    {
        $invalidData = [
            'title' => '', // 必須項目が空
            'description' => str_repeat('a', 2001), // 長すぎる説明
            'category_id' => 999, // 存在しないカテゴリ
            'due_date' => '2020-01-01', // 過去の日付
            'priority' => 'invalid', // 不正な優先度
        ];

        $response = $this->actingAs($this->user)
                        ->post('/tasks', $invalidData);

        // バリデーションエラーの確認
        $response->assertSessionHasErrors([
            'title',
            'description',
            'category_id',
            'due_date',
            'priority'
        ]);
        
        // データベースに保存されていないことを確認
        $this->assertDatabaseMissing('tasks', [
            'user_id' => $this->user->id,
        ]);
    }

    /**
     * タスクの詳細表示が正常に動作することをテスト
     */
    public function test_user_can_view_own_task_details(): void
    {
        $task = Task::factory()->for($this->user)->create([
            'title' => 'テストタスク詳細表示',
            'description' => '詳細な説明文です。',
            'priority' => 'medium',
        ]);

        $response = $this->actingAs($this->user)
                        ->get("/tasks/{$task->id}");

        $response->assertStatus(200);
        $response->assertViewIs('tasks.show');
        $response->assertSee('テストタスク詳細表示');
        $response->assertSee('詳細な説明文です。');
        $response->assertSee('medium');
        
        // ビューに正しいデータが渡されていることを確認
        $response->assertViewHas('task', function ($viewTask) use ($task) {
            return $viewTask->id === $task->id;
        });
    }

    /**
     * 他ユーザーのタスク詳細にアクセスできないことをテスト
     */
    public function test_user_cannot_view_other_users_task(): void
    {
        $task = Task::factory()->for($this->otherUser)->create();

        $response = $this->actingAs($this->user)
                        ->get("/tasks/{$task->id}");

        $response->assertStatus(403); // Forbidden
    }

    /**
     * 存在しないタスクへのアクセスで404エラーになることをテスト
     */
    public function test_viewing_nonexistent_task_returns_404(): void
    {
        $response = $this->actingAs($this->user)
                        ->get('/tasks/99999');

        $response->assertStatus(404);
    }

    /**
     * タスクの更新が正常に動作することをテスト
     */
    public function test_user_can_update_own_task(): void
    {
        $task = Task::factory()->for($this->user)->create([
            'title' => '元のタイトル',
            'description' => '元の説明',
            'priority' => 'low',
            'completed' => false,
        ]);

        $updateData = [
            'title' => '更新されたタイトル',
            'description' => '更新された説明文',
            'category_id' => $this->category->id,
            'due_date' => now()->addDays(14)->format('Y-m-d'),
            'priority' => 'high',
            'completed' => true,
            'is_private' => false,
        ];

        $response = $this->actingAs($this->user)
                        ->put("/tasks/{$task->id}", $updateData);

        // リダイレクトの確認
        $response->assertRedirect("/tasks/{$task->id}");
        
        // データベースの更新確認
        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'title' => '更新されたタイトル',
            'description' => '更新された説明文',
            'priority' => 'high',
            'completed' => true,
            'is_private' => false,
        ]);

        // updated_at が更新されていることを確認
        $updatedTask = Task::find($task->id);
        $this->assertTrue($updatedTask->updated_at->gt($task->updated_at));
        
        // 成功メッセージの確認
        $response->assertSessionHas('success');
    }

    /**
     * 他ユーザーのタスクを更新できないことをテスト
     */
    public function test_user_cannot_update_other_users_task(): void
    {
        $task = Task::factory()->for($this->otherUser)->create();

        $response = $this->actingAs($this->user)
                        ->put("/tasks/{$task->id}", [
                            'title' => '不正な更新試行',
                        ]);

        $response->assertStatus(403); // Forbidden
        
        // 元のデータが変更されていないことを確認
        $task->refresh();
        $this->assertNotEquals('不正な更新試行', $task->title);
    }

    /**
     * タスクの削除が正常に動作することをテスト
     */
    public function test_user_can_delete_own_task(): void
    {
        $task = Task::factory()->for($this->user)->create([
            'title' => '削除対象タスク',
        ]);

        $taskId = $task->id;

        $response = $this->actingAs($this->user)
                        ->delete("/tasks/{$taskId}");

        $response->assertRedirect('/tasks');
        
        // ソフトデリートの確認
        $this->assertSoftDeleted('tasks', [
            'id' => $taskId,
        ]);

        // 物理的にはまだ存在することを確認
        $this->assertDatabaseHas('tasks', [
            'id' => $taskId,
        ]);
        
        // deleted_at が設定されていることを確認
        $deletedTask = Task::withTrashed()->find($taskId);
        $this->assertNotNull($deletedTask->deleted_at);
        
        // 成功メッセージの確認
        $response->assertSessionHas('success');
    }

    /**
     * 他ユーザーのタスクを削除できないことをテスト
     */
    public function test_user_cannot_delete_other_users_task(): void
    {
        $task = Task::factory()->for($this->otherUser)->create();

        $response = $this->actingAs($this->user)
                        ->delete("/tasks/{$task->id}");

        $response->assertStatus(403); // Forbidden
        
        // タスクが削除されていないことを確認
        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'deleted_at' => null,
        ]);
    }

    /**
     * タスクの検索機能が正常に動作することをテスト
     */
    public function test_user_can_search_tasks(): void
    {
        // 検索対象のタスクを作成
        Task::factory()->for($this->user)->create(['title' => 'Laravel 学習計画']);
        Task::factory()->for($this->user)->create(['title' => 'PHP 開発タスク']);
        Task::factory()->for($this->user)->create(['title' => 'JavaScript 勉強']);
        Task::factory()->for($this->user)->create(['title' => '会議資料作成']);

        // Laravel で検索
        $response = $this->actingAs($this->user)
                        ->get('/tasks?search=Laravel');

        $response->assertStatus(200);
        $response->assertSee('Laravel 学習計画');
        $response->assertDontSee('PHP 開発タスク');
        $response->assertDontSee('JavaScript 勉強');
        $response->assertDontSee('会議資料作成');

        // 部分一致検索のテスト
        $response = $this->actingAs($this->user)
                        ->get('/tasks?search=開発');

        $response->assertStatus(200);
        $response->assertSee('PHP 開発タスク');
        $response->assertDontSee('Laravel 学習計画');
    }

    /**
     * タスクのステータスフィルタリングが正常に動作することをテスト
     */
    public function test_user_can_filter_tasks_by_completion_status(): void
    {
        // 完了済みタスクと未完了タスクを作成
        Task::factory()->for($this->user)->create([
            'completed' => true,
            'title' => '完了済みタスク1'
        ]);
        Task::factory()->for($this->user)->create([
            'completed' => true,
            'title' => '完了済みタスク2'
        ]);
        Task::factory()->for($this->user)->create([
            'completed' => false,
            'title' => '未完了タスク1'
        ]);
        Task::factory()->for($this->user)->create([
            'completed' => false,
            'title' => '未完了タスク2'
        ]);

        // 完了済みタスクのフィルタ
        $response = $this->actingAs($this->user)
                        ->get('/tasks?completed=1');

        $response->assertStatus(200);
        $response->assertSee('完了済みタスク1');
        $response->assertSee('完了済みタスク2');
        $response->assertDontSee('未完了タスク1');
        $response->assertDontSee('未完了タスク2');

        // 未完了タスクのフィルタ
        $response = $this->actingAs($this->user)
                        ->get('/tasks?completed=0');

        $response->assertStatus(200);
        $response->assertSee('未完了タスク1');
        $response->assertSee('未完了タスク2');
        $response->assertDontSee('完了済みタスク1');
        $response->assertDontSee('完了済みタスク2');
    }

    /**
     * 優先度によるフィルタリングが正常に動作することをテスト
     */
    public function test_user_can_filter_tasks_by_priority(): void
    {
        Task::factory()->for($this->user)->create(['priority' => 'high', 'title' => '緊急タスク']);
        Task::factory()->for($this->user)->create(['priority' => 'medium', 'title' => '通常タスク']);
        Task::factory()->for($this->user)->create(['priority' => 'low', 'title' => '低優先度タスク']);

        // 高優先度フィルタ
        $response = $this->actingAs($this->user)
                        ->get('/tasks?priority=high');

        $response->assertStatus(200);
        $response->assertSee('緊急タスク');
        $response->assertDontSee('通常タスク');
        $response->assertDontSee('低優先度タスク');
    }

    /**
     * タスクの一括操作が正常に動作することをテスト
     */
    public function test_user_can_perform_bulk_actions_on_tasks(): void
    {
        $tasks = Task::factory(5)->for($this->user)->create(['completed' => false]);
        $taskIds = $tasks->pluck('id')->toArray();

        // 一括完了操作
        $response = $this->actingAs($this->user)
                        ->post('/tasks/bulk-action', [
                            'task_ids' => $taskIds,
                            'action' => 'complete',
                        ]);

        $response->assertRedirect('/tasks');
        $response->assertSessionHas('success');

        // すべてのタスクが完了状態になっていることを確認
        foreach ($taskIds as $taskId) {
            $this->assertDatabaseHas('tasks', [
                'id' => $taskId,
                'completed' => true,
            ]);
        }

        // 一括削除操作のテスト
        $newTasks = Task::factory(3)->for($this->user)->create();
        $newTaskIds = $newTasks->pluck('id')->toArray();

        $response = $this->actingAs($this->user)
                        ->post('/tasks/bulk-action', [
                            'task_ids' => $newTaskIds,
                            'action' => 'delete',
                        ]);

        $response->assertRedirect('/tasks');

        // すべてのタスクが削除されていることを確認
        foreach ($newTaskIds as $taskId) {
            $this->assertSoftDeleted('tasks', [
                'id' => $taskId,
            ]);
        }
    }

    /**
     * 一括操作で他ユーザーのタスクが操作できないことをテスト
     */
    public function test_bulk_actions_cannot_affect_other_users_tasks(): void
    {
        $userTasks = Task::factory(2)->for($this->user)->create();
        $otherUserTasks = Task::factory(2)->for($this->otherUser)->create();

        $allTaskIds = $userTasks->concat($otherUserTasks)->pluck('id')->toArray();

        $response = $this->actingAs($this->user)
                        ->post('/tasks/bulk-action', [
                            'task_ids' => $allTaskIds,
                            'action' => 'complete',
                        ]);

        // 自分のタスクのみ更新されていることを確認
        foreach ($userTasks as $task) {
            $this->assertDatabaseHas('tasks', [
                'id' => $task->id,
                'completed' => true, // 更新されている
            ]);
        }

        // 他ユーザーのタスクは更新されていないことを確認
        foreach ($otherUserTasks as $task) {
            $this->assertDatabaseHas('tasks', [
                'id' => $task->id,
                'completed' => false, // 更新されていない
            ]);
        }
    }

    /**
     * ページネーションが正常に動作することをテスト
     */
    public function test_task_pagination_works_correctly(): void
    {
        // 20個のタスクを作成（ページング設定は15個/ページ）
        Task::factory(20)->for($this->user)->create();

        // 1ページ目
        $response = $this->actingAs($this->user)->get('/tasks');
        $response->assertStatus(200);
        
        // ページネーションリンクが表示されることを確認
        $response->assertSee('次へ'); // "Next" の日本語版
        
        // 2ページ目
        $response = $this->actingAs($this->user)->get('/tasks?page=2');
        $response->assertStatus(200);
        $response->assertSee('前へ'); // "Previous" の日本語版

        // ページネーションのメタデータ確認
        $response->assertViewHas('tasks', function ($tasks) {
            return $tasks instanceof \Illuminate\Pagination\LengthAwarePaginator;
        });
    }

    /**
     * ソート機能が正常に動作することをテスト
     */
    public function test_task_sorting_works_correctly(): void
    {
        // 異なる作成日時でタスクを作成
        $oldTask = Task::factory()->for($this->user)->create([
            'title' => '古いタスク',
            'created_at' => now()->subDays(5),
        ]);
        
        $newTask = Task::factory()->for($this->user)->create([
            'title' => '新しいタスク',
            'created_at' => now(),
        ]);

        // 新しい順（デフォルト）
        $response = $this->actingAs($this->user)->get('/tasks');
        $content = $response->getContent();
        $newPos = strpos($content, '新しいタスク');
        $oldPos = strpos($content, '古いタスク');
        $this->assertTrue($newPos < $oldPos);

        // 古い順
        $response = $this->actingAs($this->user)
                        ->get('/tasks?sort=created_at&direction=asc');
        $content = $response->getContent();
        $newPos = strpos($content, '新しいタスク');
        $oldPos = strpos($content, '古いタスク');
        $this->assertTrue($oldPos < $newPos);
    }

    /**
     * ダッシュボードでの統計情報が正しく表示されることをテスト
     */
    public function test_dashboard_shows_correct_task_statistics(): void
    {
        // 統計用のタスクを作成
        Task::factory(5)->for($this->user)->create(['completed' => true]);
        Task::factory(3)->for($this->user)->create(['completed' => false]);
        Task::factory(2)->for($this->user)->create([
            'completed' => false,
            'due_date' => now()->addDays(3), // 近日期限
        ]);

        $response = $this->actingAs($this->user)->get('/dashboard');

        $response->assertStatus(200);
        $response->assertViewIs('dashboard');

        // 統計情報の確認
        $response->assertViewHas('stats', function ($stats) {
            return $stats['total_tasks'] === 10 &&
                   $stats['completed_tasks'] === 5 &&
                   $stats['pending_tasks'] === 5 &&
                   $stats['completion_rate'] === 50.0;
        });
    }

    /**
     * タスクステータスのAjax切り替えが正常に動作することをテスト
     */
    public function test_task_status_can_be_toggled_via_ajax(): void
    {
        $task = Task::factory()->for($this->user)->create(['completed' => false]);

        // Ajax リクエストでステータス切り替え
        $response = $this->actingAs($this->user)
                        ->postJson("/tasks/{$task->id}/toggle-status", [
                            'completed' => true,
                        ]);

        $response->assertStatus(200)
                ->assertJson([
                    'success' => true,
                    'completed' => true,
                ]);

        // データベースが更新されていることを確認
        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'completed' => true,
        ]);
    }

    /**
     * 無効なデータでのAjaxリクエストが適切にエラーを返すことをテスト
     */
    public function test_ajax_requests_validate_data_properly(): void
    {
        $task = Task::factory()->for($this->user)->create();

        // 無効なデータでリクエスト
        $response = $this->actingAs($this->user)
                        ->postJson("/tasks/{$task->id}/toggle-status", [
                            'completed' => 'invalid',
                        ]);

        $response->assertStatus(422); // Validation error
    }
}